<?php

declare(strict_types=1);

namespace OCA\Talk\Vendor\CuyZ\Valinor\Type\Types;

use OCA\Talk\Vendor\CuyZ\Valinor\Mapper\Tree\Message\ErrorMessage;
use OCA\Talk\Vendor\CuyZ\Valinor\Mapper\Tree\Message\MessageBuilder;
use OCA\Talk\Vendor\CuyZ\Valinor\Type\FixedType;
use OCA\Talk\Vendor\CuyZ\Valinor\Type\FloatType;
use OCA\Talk\Vendor\CuyZ\Valinor\Type\Type;

use function assert;

/** @internal */
final class FloatValueType implements FloatType, FixedType
{
    public function __construct(private float $value) {}

    public function accepts(mixed $value): bool
    {
        return $value === $this->value;
    }

    public function matches(Type $other): bool
    {
        if ($other instanceof UnionType) {
            return $other->isMatchedBy($this);
        }

        if ($other instanceof self) {
            return $this->value === $other->value;
        }

        return $other instanceof NativeFloatType || $other instanceof MixedType;
    }

    public function canCast(mixed $value): bool
    {
        return is_numeric($value) && (float)$value === $this->value;
    }

    public function cast(mixed $value): float
    {
        assert($this->canCast($value));

        return (float)$value; // @phpstan-ignore-line
    }

    public function errorMessage(): ErrorMessage
    {
        return MessageBuilder::newError('Value {source_value} does not match float value {expected_value}.')
            ->withParameter('expected_value', (string)$this->value)
            ->build();
    }

    public function value(): float
    {
        return $this->value;
    }

    public function toString(): string
    {
        return (string)$this->value;
    }
}
