"""
Copyright (c) 2025 Proton AG

This file is part of Proton VPN.

Proton VPN is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Proton VPN is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with ProtonVPN.  If not, see <https://www.gnu.org/licenses/>.
"""
from __future__ import annotations
from typing import Union

from dataclasses import dataclass, asdict

from gi.repository import Gtk, GObject, GdkPixbuf, GLib


from proton.vpn.app.gtk.widgets.headerbar.menu.settings.common import \
    SettingName
from proton.vpn.app.gtk.assets import icons


ICON_SIZE_IN_PX = 16


@dataclass(eq=True, frozen=True)
class AppData:  # pylint: disable=missing-class-docstring
    name: str
    executable: str
    icon_name: str
    native: bool

    def to_dict(self) -> dict[str, Union[str, bool]]:
        """Convert dataclass to dict

        Returns:
            dict[str, Union[str, bool]]
        """
        return asdict(self)

    @staticmethod
    def from_dict(data: dict[str, Union[str, bool]]) -> AppData:
        """Build object based on dict.

        Args:
            data (dict[str, Union[str, bool]])

        Returns:
            AppData
        """
        return AppData(
            name=data["name"],
            executable=data["executable"],
            icon_name=data["icon_name"],
            native=data["native"]
        )


def _get_missing_icon_pixbuff() -> GdkPixbuf.Pixbuf:
    return icons.get("no-app-icon.svg")


def get_icon(img_path: Union[str, None], gtk: Gtk = Gtk) -> Gtk.Image:
    """Returns a Gtk.Image based either on the app path image or else
    uses a default one.

    Args:
        img_path (str)
        gtk (Gtk, optional): Defaults to Gtk.

    Returns:
        Gtk.Image
    """
    # If it starts with / then we've received a path to an image
    pixbuff = None

    if not img_path:
        pixbuff = _get_missing_icon_pixbuff()
    elif img_path.startswith("/"):
        pixbuff = GdkPixbuf.Pixbuf.new_from_file_at_scale(
            filename=img_path,
            width=ICON_SIZE_IN_PX,
            height=-1,
            preserve_aspect_ratio=True
        )
    else:
        theme = Gtk.IconTheme.get_default()
        try:
            # This can still return None if the object is not found
            pixbuff = theme.load_icon(img_path, ICON_SIZE_IN_PX, Gtk.IconLookupFlags.FORCE_SIZE)

        # We don't want to crash if for some reason it's impossible to load the icon.
        # Since it's not a full filepath we can not either check if it exists or not.
        except GLib.Error:
            pass

    if not pixbuff:
        pixbuff = _get_missing_icon_pixbuff()

    return gtk.Image.new_from_pixbuf(pixbuff)


class AppRowWithCheckbox(Gtk.Grid):
    """App row used to display when selecting apps to be split tunneled.
    """
    def __init__(
        self,
        app_data: AppData,
        checked: bool,
        gtk: Gtk = Gtk
    ):
        super().__init__()
        self.set_column_spacing(10)
        self.set_name(app_data.executable)

        self.app_data = app_data
        self._checked = checked
        self.gtk = gtk
        self._check_button = None

    @staticmethod
    def build(app_data: AppData, checked: bool) -> AppRowWithCheckbox:
        """Method to automate building the object

        Args:
            app_data (AppData):
            checked (bool)

        Returns:
            AppRowWithCheckbox
        """
        app_row = AppRowWithCheckbox(app_data, checked)
        app_row.build_ui()

        return app_row

    def build_ui(self):
        """Build the UI.
        It's still the parents responsibility to display it.
        """
        icon = get_icon(self.app_data.icon_name, self.gtk)
        label = SettingName(self.app_data.name)

        self._check_button = self.gtk.CheckButton.new()
        self._check_button.set_active(self._checked)

        self.attach(self._check_button, 0, 0, 1, 1)
        self.attach(icon, 1, 0, 1, 1)
        self.attach(label, 2, 0, 1, 1)

    @property
    def checked(self) -> bool:
        """Only available when built with `AppRow.build_with_remove_button()`.

        Returns:
            bool
        """
        return self._check_button.get_active()

    def _set_check(self, val: bool):
        """Mainly for testing purposes and not for public API.

        Args:
            val (bool)
        """
        self._check_button.set_active(val)


class AppRowWithRemoveButton(Gtk.Grid):
    """_summary_
    """
    def __init__(self, app_data: AppData, gtk: Gtk = Gtk):
        super().__init__()
        self.set_column_spacing(10)
        self.set_name(app_data.executable)

        self.app_data = app_data
        self.gtk = gtk
        self._remove_button = None

    @staticmethod
    def build(app_data: AppData) -> AppRowWithRemoveButton:
        """Method to automate building the object

        Args:
            app_data (AppData)

        Returns:
            AppRowWithRemoveButton
        """
        app_row = AppRowWithRemoveButton(app_data)
        app_row.build_ui()

        return app_row

    def build_ui(self):
        """Build the UI.
        It's still the parents responsibility to display it.
        """
        icon = get_icon(self.app_data.icon_name, self.gtk)
        label = SettingName(self.app_data.name)
        self._remove_button = Gtk.Button.new_from_icon_name("edit-delete-symbolic", 1)
        self._remove_button.connect("clicked", self._signal_remove_app)

        self.attach(icon, 0, 0, 1, 1)
        self.attach(label, 1, 0, 1, 1)
        self.attach(self._remove_button, 2, 0, 1, 1)

    def _signal_remove_app(self, _: Gtk.Button):
        self.emit("remove-app")

    @GObject.Signal(name="remove-app")
    def remove_app(self):
        """
        Signal emitted after the user clicks on remove button.

        Since this object returns itself, we don't need to pass any arguments
        as we can easily access the `app` property.
        """

    def _click_on_remove_button(self):
        """Mainly for testing purposes and not for public API.
        """
        self._remove_button.clicked()
