"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.nodeKdf = exports.curryCryptoStream = exports.getDecryptionHelper = exports.getEncryptHelper = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const crypto_1 = require("crypto");
const hkdf_node_1 = require("@aws-crypto/hkdf-node");
const serialize_1 = require("@aws-crypto/serialize");
const kdfIndex = Object.freeze({
    sha256: (0, hkdf_node_1.HKDF)('sha256'),
    sha384: (0, hkdf_node_1.HKDF)('sha384'),
    sha512: (0, hkdf_node_1.HKDF)('sha512'),
});
const getEncryptHelper = (material) => {
    /* Precondition: NodeEncryptionMaterial must have a valid data key. */
    (0, material_management_1.needs)(material.hasValidKey(), 'Material has no unencrypted data key.');
    const { signatureHash } = material.suite;
    /* Conditional types can not narrow the return type :(
     * Function overloads "works" but then I can not export
     * the function and have eslint be happy (Multiple exports of name)
     */
    const getCipherInfo = curryCryptoStream(material, crypto_1.createCipheriv);
    return Object.freeze({
        getCipherInfo,
        getSigner: signatureHash ? getSigner : undefined,
        dispose,
    });
    function getSigner() {
        /* Precondition: The NodeEncryptionMaterial must have not been zeroed.
         * hasUnencryptedDataKey will check that the unencrypted data key has been set
         * *and* that it has not been zeroed.  At this point it must have been set
         * because the KDF function operated on it.  So at this point
         * we are protecting that someone has zeroed out the material
         * because the Encrypt process has been complete.
         */
        (0, material_management_1.needs)(material.hasUnencryptedDataKey, 'Unencrypted data key has been zeroed.');
        if (!signatureHash)
            throw new Error('Material does not support signature.');
        const { signatureKey } = material;
        if (!signatureKey)
            throw new Error('Material does not support signature.');
        const { privateKey } = signatureKey;
        if (typeof privateKey !== 'string')
            throw new Error('Material does not support signature.');
        const signer = Object.assign((0, crypto_1.createSign)(signatureHash), 
        // don't export the private key if we don't have to
        { awsCryptoSign: () => signer.sign(privateKey) });
        return signer;
    }
    function dispose() {
        material.zeroUnencryptedDataKey();
    }
};
exports.getEncryptHelper = getEncryptHelper;
const getDecryptionHelper = (material) => {
    /* Precondition: NodeDecryptionMaterial must have a valid data key. */
    (0, material_management_1.needs)(material.hasValidKey(), 'Material has no unencrypted data key.');
    const { signatureHash } = material.suite;
    /* Conditional types can not narrow the return type :(
     * Function overloads "works" but then I can not export
     * the function and have eslint be happy (Multiple exports of name)
     */
    const getDecipherInfo = curryCryptoStream(material, crypto_1.createDecipheriv);
    return Object.freeze({
        getDecipherInfo,
        getVerify: signatureHash ? getVerify : undefined,
        dispose,
    });
    function getVerify() {
        if (!signatureHash)
            throw new Error('Material does not support signature.');
        const { verificationKey } = material;
        if (!verificationKey)
            throw new Error('Material does not support signature.');
        const verify = Object.assign((0, crypto_1.createVerify)(signatureHash), 
        // explicitly bind the public key for this material
        {
            awsCryptoVerify: (signature) => 
            // As typescript gets better typing
            // We should consider either generics or
            // 2 different verificationKeys for Node and WebCrypto
            verify.verify(verificationKey.publicKey, signature),
        });
        return verify;
    }
    function dispose() {
        material.zeroUnencryptedDataKey();
    }
};
exports.getDecryptionHelper = getDecryptionHelper;
function curryCryptoStream(material, createCryptoIvStream) {
    const { encryption: cipherName, ivLength } = material.suite;
    const isEncrypt = material instanceof material_management_1.NodeEncryptionMaterial;
    /* Precondition: material must be either NodeEncryptionMaterial or NodeDecryptionMaterial.
     *
     */
    (0, material_management_1.needs)(isEncrypt
        ? crypto_1.createCipheriv === createCryptoIvStream
        : material instanceof material_management_1.NodeDecryptionMaterial
            ? crypto_1.createDecipheriv === createCryptoIvStream
            : false, 'Unsupported cryptographic material.');
    return (messageId, commitKey) => {
        const { derivedKey, keyCommitment } = nodeKdf(material, messageId, commitKey);
        return (isEncrypt
            ? { getCipher: createCryptoStream, keyCommitment }
            : createCryptoStream);
        function createCryptoStream(iv) {
            /* Precondition: The length of the IV must match the NodeAlgorithmSuite specification. */
            (0, material_management_1.needs)(iv.byteLength === ivLength, 'Iv length does not match algorithm suite specification');
            /* Precondition: The material must have not been zeroed.
             * hasUnencryptedDataKey will check that the unencrypted data key has been set
             * *and* that it has not been zeroed.  At this point it must have been set
             * because the KDF function operated on it.  So at this point
             * we are protecting that someone has zeroed out the material
             * because the Encrypt process has been complete.
             */
            (0, material_management_1.needs)(material.hasUnencryptedDataKey, 'Unencrypted data key has been zeroed.');
            /* createDecipheriv is incorrectly typed in @types/node. It should take key: CipherKey, not key: BinaryLike.
             * Also, the check above ensures
             * that _createCryptoStream is not false.
             * But TypeScript does not believe me.
             * For any complicated code,
             * you should defer to the checker,
             * but here I'm going to assert
             * it is simple enough.
             */
            return createCryptoIvStream(cipherName, derivedKey, iv);
        }
    };
}
exports.curryCryptoStream = curryCryptoStream;
function nodeKdf(material, nonce, commitKey) {
    const dataKey = material.getUnencryptedDataKey();
    const { kdf, kdfHash, keyLengthBytes, commitmentLength, saltLengthBytes, commitment, id: suiteId, } = material.suite;
    /* Check for early return (Postcondition): No Node.js KDF, just return the unencrypted data key. */
    if (!kdf && !kdfHash) {
        /* Postcondition: Non-KDF algorithm suites *must* not have a commitment. */
        (0, material_management_1.needs)(!commitKey, 'Commitment not supported.');
        return { derivedKey: dataKey };
    }
    /* Precondition: Valid HKDF values must exist for Node.js. */
    (0, material_management_1.needs)(kdf === 'HKDF' &&
        kdfHash &&
        kdfIndex[kdfHash] &&
        nonce instanceof Uint8Array, 'Invalid HKDF values.');
    /* The unwrap is done once we *know* that a KDF is required.
     * If we unwrapped before everything will work,
     * but we may be creating new copies of the unencrypted data key (export).
     */
    const { buffer: dkBuffer, byteOffset: dkByteOffset, byteLength: dkByteLength, } = (0, material_management_1.unwrapDataKey)(dataKey);
    if (commitment === 'NONE') {
        /* Postcondition: Non-committing Node algorithm suites *must* not have a commitment. */
        (0, material_management_1.needs)(!commitKey, 'Commitment not supported.');
        const toExtract = Buffer.from(dkBuffer, dkByteOffset, dkByteLength);
        const { buffer, byteOffset, byteLength } = (0, serialize_1.kdfInfo)(suiteId, nonce);
        const infoBuff = Buffer.from(buffer, byteOffset, byteLength);
        const derivedBytes = kdfIndex[kdfHash](toExtract)(keyLengthBytes, infoBuff);
        const derivedKey = (0, material_management_1.wrapWithKeyObjectIfSupported)(derivedBytes);
        return { derivedKey };
    }
    /* Precondition UNTESTED: Committing suites must define expected values. */
    (0, material_management_1.needs)(commitment === 'KEY' && commitmentLength && saltLengthBytes, 'Malformed suite data.');
    /* Precondition: For committing algorithms, the nonce *must* be 256 bit.
     * i.e. It must target a V2 message format.
     */
    (0, material_management_1.needs)(nonce.byteLength === saltLengthBytes, 'Nonce is not the correct length for committed algorithm suite.');
    const toExtract = Buffer.from(dkBuffer, dkByteOffset, dkByteLength);
    const expand = kdfIndex[kdfHash](toExtract, nonce);
    const { keyLabel, commitLabel } = (0, serialize_1.kdfCommitKeyInfo)(material.suite);
    const keyCommitment = expand(commitmentLength / 8, commitLabel);
    const isDecrypt = material instanceof material_management_1.NodeDecryptionMaterial;
    /* Precondition: If material is NodeDecryptionMaterial the key commitments *must* match.
     * This is also the preferred location to check,
     * because then the decryption key is never even derived.
     */
    (0, material_management_1.needs)((isDecrypt && commitKey && (0, crypto_1.timingSafeEqual)(keyCommitment, commitKey)) ||
        (!isDecrypt && !commitKey), isDecrypt ? 'Commitment does not match.' : 'Invalid arguments.');
    const derivedBytes = expand(keyLengthBytes, keyLabel);
    const derivedKey = (0, material_management_1.wrapWithKeyObjectIfSupported)(derivedBytes);
    return { derivedKey, keyCommitment };
}
exports.nodeKdf = nodeKdf;
//# sourceMappingURL=data:application/json;base64,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