"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebCryptoAlgorithmSuite = void 0;
/*
 * This file contains information about particular algorithm suites used
 * within the encryption SDK.  In most cases, end-users don't need to
 * manipulate this structure, but it can occasionally be needed for more
 * advanced use cases, such as writing keyrings.
 *
 * These are the WebCrypto specific values the AWS Encryption SDK for JavaScript
 * Algorithm Suites.
 */
const algorithm_suites_1 = require("./algorithm_suites");
const needs_1 = require("./needs");
/* References to https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * These are the composed parameters for each algorithm suite specification for
 * for the WebCrypto environment.
 */
const webCryptoAlgAes128GcmIv12Tag16 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
/* Web browsers do not support 192 bit key lengths at this time. */
const webCryptoAlgAes192GcmIv12Tag16 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmIv12Tag16 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const webCryptoAlgAes128GcmIv12Tag16HkdfSha256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    commitment: 'NONE',
};
/* Web browsers do not support 192 bit key lengths at this time. */
const webCryptoAlgAes192GcmIv12Tag16HkdfSha256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmIv12Tag16HkdfSha256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    commitment: 'NONE',
};
const webCryptoAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    signatureCurve: 'P-256',
    signatureHash: 'SHA-256',
    commitment: 'NONE',
};
/* Web browsers do not support 192 bit key lengths at this time. */
const webCryptoAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-384',
    cacheSafe: true,
    signatureCurve: 'P-384',
    signatureHash: 'SHA-384',
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-384',
    cacheSafe: true,
    signatureCurve: 'P-384',
    signatureHash: 'SHA-384',
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmHkdfSha512Committing = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY,
    messageFormat: algorithm_suites_1.MessageFormat.V2,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-512',
    cacheSafe: true,
    commitment: 'KEY',
    commitmentHash: 'SHA-512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const webCryptoAlgAes256GcmHkdfSha512CommittingEcdsaP384 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    messageFormat: algorithm_suites_1.MessageFormat.V2,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-512',
    cacheSafe: true,
    signatureCurve: 'P-384',
    signatureHash: 'SHA-384',
    commitment: 'KEY',
    commitmentHash: 'SHA-512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const webCryptoAlgorithms = Object.freeze({
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512Committing),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512CommittingEcdsaP384),
});
const supportedWebCryptoAlgorithms = Object.freeze({
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16),
    // [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256),
    // [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256),
    // [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512Committing),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512CommittingEcdsaP384),
});
class WebCryptoAlgorithmSuite extends algorithm_suites_1.AlgorithmSuite {
    type = 'webCrypto';
    constructor(id) {
        super(webCryptoAlgorithms[id]);
        /* Precondition: Browsers do not support 192 bit keys so the AlgorithmSuiteIdentifier is removed.
         * This is primarily an error in decrypt but this make it clear.
         * The error can manifest deep in the decrypt loop making it hard to debug.
         */
        (0, needs_1.needs)(Object.prototype.hasOwnProperty.call(supportedWebCryptoAlgorithms, id), '192-bit AES keys are not supported');
        Object.setPrototypeOf(this, WebCryptoAlgorithmSuite.prototype);
        Object.freeze(this);
    }
}
exports.WebCryptoAlgorithmSuite = WebCryptoAlgorithmSuite;
Object.freeze(WebCryptoAlgorithmSuite.prototype);
Object.freeze(WebCryptoAlgorithmSuite);
//# sourceMappingURL=data:application/json;base64,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