"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NDJSONProcessor = void 0;
var _ndjson = require("ndjson");
var _stream = require("stream");
var _util = require("../utils/util");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

class NDJSONProcessor {
  async validateText(text, _) {
    const stringStream = new _stream.Readable();
    stringStream._read = () => {};
    stringStream.push(text);
    stringStream.push(null);
    return await new Promise((promise, reject) => {
      stringStream.pipe((0, _ndjson.parse)({
        strict: true
      })).on('error', e => reject(e)).on('data', document => {
        if (!(0, _util.isValidObject)(document)) {
          reject(new Error(`The following document has empty fields: ${JSON.stringify(document)}`));
        }
      }).on('end', () => promise(true));
    });
  }
  async ingestText(text, options) {
    const {
      client,
      indexName
    } = options;
    const stringStream = new _stream.Readable();
    stringStream._read = () => {};
    stringStream.push(text);
    stringStream.push(null);
    const failedRows = [];
    const numDocuments = await new Promise(promise => {
      const tasks = [];
      let numDocumentsCount = 0;
      stringStream.pipe((0, _ndjson.parse)({
        strict: true
      })).on('error', _ => {
        const curRow = ++numDocumentsCount;
        failedRows.push(curRow);
      }).on('data', document => {
        const task = (async () => {
          const curRow = ++numDocumentsCount;
          try {
            await client.index({
              index: indexName,
              body: document
            });
          } catch (e) {
            failedRows.push(curRow);
          }
        })();
        tasks.push(task);
      }).on('end', async () => {
        await Promise.all(tasks);
        promise(numDocumentsCount);
      });
    });
    return {
      total: numDocuments,
      message: `Indexed ${numDocuments - failedRows.length} documents`,
      failedRows: failedRows.sort((n1, n2) => n1 - n2)
    };
  }
  async ingestFile(file, options) {
    const {
      client,
      indexName
    } = options;
    const failedRows = [];
    const numDocuments = await new Promise((resolve, reject) => {
      const tasks = [];
      let numDocumentsCount = 0;
      file.pipe((0, _ndjson.parse)({
        strict: true
      })).on('error', e => reject(new Error(`Stopped processing after ${numDocumentsCount} rows due to: ${e}`))).on('data', document => {
        const task = (async () => {
          const curRow = ++numDocumentsCount;
          if (!(0, _util.isValidObject)(document)) {
            failedRows.push(curRow);
          } else {
            try {
              await client.index({
                index: indexName,
                body: document
              });
            } catch (_) {
              failedRows.push(curRow);
            }
          }
        })();
        tasks.push(task);
      }).on('end', async () => {
        await Promise.all(tasks);
        resolve(numDocumentsCount);
      });
    });
    return {
      total: numDocuments,
      message: `Indexed ${numDocuments - failedRows.length} documents`,
      failedRows: failedRows.sort((n1, n2) => n1 - n2)
    };
  }
  async parseFile(file, limit, _) {
    const documents = [];
    await new Promise((resolve, reject) => {
      file.pipe((0, _ndjson.parse)({
        strict: true
      })).on('error', e => reject(e)).on('data', document => {
        if (!(0, _util.isValidObject)(document)) {
          reject(new Error(`The following document has empty fields: ${JSON.stringify(document)}`));
        }
        if (documents.length >= limit) {
          resolve();
          file.destroy();
          return;
        }
        documents.push(document);
      }).on('end', () => {
        resolve();
      });
    });
    return documents;
  }
}
exports.NDJSONProcessor = NDJSONProcessor;