﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-messaging/ChimeSDKMessagingRequest.h>
#include <aws/chime-sdk-messaging/ChimeSDKMessaging_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ChimeSDKMessaging {
namespace Model {

/**
 */
class RedactChannelMessageRequest : public ChimeSDKMessagingRequest {
 public:
  AWS_CHIMESDKMESSAGING_API RedactChannelMessageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RedactChannelMessage"; }

  AWS_CHIMESDKMESSAGING_API Aws::String SerializePayload() const override;

  AWS_CHIMESDKMESSAGING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the channel containing the messages that you want to redact.</p>
   */
  inline const Aws::String& GetChannelArn() const { return m_channelArn; }
  inline bool ChannelArnHasBeenSet() const { return m_channelArnHasBeenSet; }
  template <typename ChannelArnT = Aws::String>
  void SetChannelArn(ChannelArnT&& value) {
    m_channelArnHasBeenSet = true;
    m_channelArn = std::forward<ChannelArnT>(value);
  }
  template <typename ChannelArnT = Aws::String>
  RedactChannelMessageRequest& WithChannelArn(ChannelArnT&& value) {
    SetChannelArn(std::forward<ChannelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the message being redacted.</p>
   */
  inline const Aws::String& GetMessageId() const { return m_messageId; }
  inline bool MessageIdHasBeenSet() const { return m_messageIdHasBeenSet; }
  template <typename MessageIdT = Aws::String>
  void SetMessageId(MessageIdT&& value) {
    m_messageIdHasBeenSet = true;
    m_messageId = std::forward<MessageIdT>(value);
  }
  template <typename MessageIdT = Aws::String>
  RedactChannelMessageRequest& WithMessageId(MessageIdT&& value) {
    SetMessageId(std::forward<MessageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the <code>AppInstanceUser</code> or <code>AppInstanceBot</code>
   * that makes the API call.</p>
   */
  inline const Aws::String& GetChimeBearer() const { return m_chimeBearer; }
  inline bool ChimeBearerHasBeenSet() const { return m_chimeBearerHasBeenSet; }
  template <typename ChimeBearerT = Aws::String>
  void SetChimeBearer(ChimeBearerT&& value) {
    m_chimeBearerHasBeenSet = true;
    m_chimeBearer = std::forward<ChimeBearerT>(value);
  }
  template <typename ChimeBearerT = Aws::String>
  RedactChannelMessageRequest& WithChimeBearer(ChimeBearerT&& value) {
    SetChimeBearer(std::forward<ChimeBearerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the SubChannel in the request.</p>
   */
  inline const Aws::String& GetSubChannelId() const { return m_subChannelId; }
  inline bool SubChannelIdHasBeenSet() const { return m_subChannelIdHasBeenSet; }
  template <typename SubChannelIdT = Aws::String>
  void SetSubChannelId(SubChannelIdT&& value) {
    m_subChannelIdHasBeenSet = true;
    m_subChannelId = std::forward<SubChannelIdT>(value);
  }
  template <typename SubChannelIdT = Aws::String>
  RedactChannelMessageRequest& WithSubChannelId(SubChannelIdT&& value) {
    SetSubChannelId(std::forward<SubChannelIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelArn;

  Aws::String m_messageId;

  Aws::String m_chimeBearer;

  Aws::String m_subChannelId;
  bool m_channelArnHasBeenSet = false;
  bool m_messageIdHasBeenSet = false;
  bool m_chimeBearerHasBeenSet = false;
  bool m_subChannelIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ChimeSDKMessaging
}  // namespace Aws
