﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeartifact/CodeArtifactRequest.h>
#include <aws/codeartifact/CodeArtifact_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeArtifact {
namespace Model {

/**
 */
class PutDomainPermissionsPolicyRequest : public CodeArtifactRequest {
 public:
  AWS_CODEARTIFACT_API PutDomainPermissionsPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutDomainPermissionsPolicy"; }

  AWS_CODEARTIFACT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The name of the domain on which to set the resource policy. </p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  PutDomainPermissionsPolicyRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The 12-digit account number of the Amazon Web Services account that owns the
   * domain. It does not include dashes or spaces. </p>
   */
  inline const Aws::String& GetDomainOwner() const { return m_domainOwner; }
  inline bool DomainOwnerHasBeenSet() const { return m_domainOwnerHasBeenSet; }
  template <typename DomainOwnerT = Aws::String>
  void SetDomainOwner(DomainOwnerT&& value) {
    m_domainOwnerHasBeenSet = true;
    m_domainOwner = std::forward<DomainOwnerT>(value);
  }
  template <typename DomainOwnerT = Aws::String>
  PutDomainPermissionsPolicyRequest& WithDomainOwner(DomainOwnerT&& value) {
    SetDomainOwner(std::forward<DomainOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The current revision of the resource policy to be set. This revision is used
   * for optimistic locking, which prevents others from overwriting your changes to
   * the domain's resource policy. </p>
   */
  inline const Aws::String& GetPolicyRevision() const { return m_policyRevision; }
  inline bool PolicyRevisionHasBeenSet() const { return m_policyRevisionHasBeenSet; }
  template <typename PolicyRevisionT = Aws::String>
  void SetPolicyRevision(PolicyRevisionT&& value) {
    m_policyRevisionHasBeenSet = true;
    m_policyRevision = std::forward<PolicyRevisionT>(value);
  }
  template <typename PolicyRevisionT = Aws::String>
  PutDomainPermissionsPolicyRequest& WithPolicyRevision(PolicyRevisionT&& value) {
    SetPolicyRevision(std::forward<PolicyRevisionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A valid displayable JSON Aspen policy string to be set as the access control
   * resource policy on the provided domain. </p>
   */
  inline const Aws::String& GetPolicyDocument() const { return m_policyDocument; }
  inline bool PolicyDocumentHasBeenSet() const { return m_policyDocumentHasBeenSet; }
  template <typename PolicyDocumentT = Aws::String>
  void SetPolicyDocument(PolicyDocumentT&& value) {
    m_policyDocumentHasBeenSet = true;
    m_policyDocument = std::forward<PolicyDocumentT>(value);
  }
  template <typename PolicyDocumentT = Aws::String>
  PutDomainPermissionsPolicyRequest& WithPolicyDocument(PolicyDocumentT&& value) {
    SetPolicyDocument(std::forward<PolicyDocumentT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;

  Aws::String m_domainOwner;

  Aws::String m_policyRevision;

  Aws::String m_policyDocument;
  bool m_domainHasBeenSet = false;
  bool m_domainOwnerHasBeenSet = false;
  bool m_policyRevisionHasBeenSet = false;
  bool m_policyDocumentHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeArtifact
}  // namespace Aws
