﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/TaskRunFilterCriteria.h>
#include <aws/glue/model/TaskRunSortCriteria.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class GetMLTaskRunsRequest : public GlueRequest {
 public:
  AWS_GLUE_API GetMLTaskRunsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetMLTaskRuns"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The unique identifier of the machine learning transform.</p>
   */
  inline const Aws::String& GetTransformId() const { return m_transformId; }
  inline bool TransformIdHasBeenSet() const { return m_transformIdHasBeenSet; }
  template <typename TransformIdT = Aws::String>
  void SetTransformId(TransformIdT&& value) {
    m_transformIdHasBeenSet = true;
    m_transformId = std::forward<TransformIdT>(value);
  }
  template <typename TransformIdT = Aws::String>
  GetMLTaskRunsRequest& WithTransformId(TransformIdT&& value) {
    SetTransformId(std::forward<TransformIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token for pagination of the results. The default is empty.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetMLTaskRunsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetMLTaskRunsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filter criteria, in the <code>TaskRunFilterCriteria</code> structure, for
   * the task run.</p>
   */
  inline const TaskRunFilterCriteria& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = TaskRunFilterCriteria>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = TaskRunFilterCriteria>
  GetMLTaskRunsRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sorting criteria, in the <code>TaskRunSortCriteria</code> structure, for
   * the task run.</p>
   */
  inline const TaskRunSortCriteria& GetSort() const { return m_sort; }
  inline bool SortHasBeenSet() const { return m_sortHasBeenSet; }
  template <typename SortT = TaskRunSortCriteria>
  void SetSort(SortT&& value) {
    m_sortHasBeenSet = true;
    m_sort = std::forward<SortT>(value);
  }
  template <typename SortT = TaskRunSortCriteria>
  GetMLTaskRunsRequest& WithSort(SortT&& value) {
    SetSort(std::forward<SortT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_transformId;

  Aws::String m_nextToken;

  int m_maxResults{0};

  TaskRunFilterCriteria m_filter;

  TaskRunSortCriteria m_sort;
  bool m_transformIdHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_filterHasBeenSet = false;
  bool m_sortHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
