﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/groundstation/GroundStationRequest.h>
#include <aws/groundstation/GroundStation_EXPORTS.h>
#include <aws/groundstation/model/EphemerisData.h>

#include <utility>

namespace Aws {
namespace GroundStation {
namespace Model {

/**
 */
class CreateEphemerisRequest : public GroundStationRequest {
 public:
  AWS_GROUNDSTATION_API CreateEphemerisRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEphemeris"; }

  AWS_GROUNDSTATION_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The satellite ID that associates this ephemeris with a satellite in AWS
   * Ground Station.</p>
   */
  inline const Aws::String& GetSatelliteId() const { return m_satelliteId; }
  inline bool SatelliteIdHasBeenSet() const { return m_satelliteIdHasBeenSet; }
  template <typename SatelliteIdT = Aws::String>
  void SetSatelliteId(SatelliteIdT&& value) {
    m_satelliteIdHasBeenSet = true;
    m_satelliteId = std::forward<SatelliteIdT>(value);
  }
  template <typename SatelliteIdT = Aws::String>
  CreateEphemerisRequest& WithSatelliteId(SatelliteIdT&& value) {
    SetSatelliteId(std::forward<SatelliteIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> to enable the ephemeris after validation. Set to
   * <code>false</code> to keep it disabled.</p>
   */
  inline bool GetEnabled() const { return m_enabled; }
  inline bool EnabledHasBeenSet() const { return m_enabledHasBeenSet; }
  inline void SetEnabled(bool value) {
    m_enabledHasBeenSet = true;
    m_enabled = value;
  }
  inline CreateEphemerisRequest& WithEnabled(bool value) {
    SetEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A priority score that determines which ephemeris to use when multiple
   * ephemerides overlap.</p> <p>Higher numbers take precedence. The default is 1.
   * Must be 1 or greater.</p>
   */
  inline int GetPriority() const { return m_priority; }
  inline bool PriorityHasBeenSet() const { return m_priorityHasBeenSet; }
  inline void SetPriority(int value) {
    m_priorityHasBeenSet = true;
    m_priority = value;
  }
  inline CreateEphemerisRequest& WithPriority(int value) {
    SetPriority(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An overall expiration time for the ephemeris in UTC, after which it will
   * become <code>EXPIRED</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetExpirationTime() const { return m_expirationTime; }
  inline bool ExpirationTimeHasBeenSet() const { return m_expirationTimeHasBeenSet; }
  template <typename ExpirationTimeT = Aws::Utils::DateTime>
  void SetExpirationTime(ExpirationTimeT&& value) {
    m_expirationTimeHasBeenSet = true;
    m_expirationTime = std::forward<ExpirationTimeT>(value);
  }
  template <typename ExpirationTimeT = Aws::Utils::DateTime>
  CreateEphemerisRequest& WithExpirationTime(ExpirationTimeT&& value) {
    SetExpirationTime(std::forward<ExpirationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name that you can use to identify the ephemeris.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateEphemerisRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the KMS key to use for encrypting the ephemeris.</p>
   */
  inline const Aws::String& GetKmsKeyArn() const { return m_kmsKeyArn; }
  inline bool KmsKeyArnHasBeenSet() const { return m_kmsKeyArnHasBeenSet; }
  template <typename KmsKeyArnT = Aws::String>
  void SetKmsKeyArn(KmsKeyArnT&& value) {
    m_kmsKeyArnHasBeenSet = true;
    m_kmsKeyArn = std::forward<KmsKeyArnT>(value);
  }
  template <typename KmsKeyArnT = Aws::String>
  CreateEphemerisRequest& WithKmsKeyArn(KmsKeyArnT&& value) {
    SetKmsKeyArn(std::forward<KmsKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Ephemeris data.</p>
   */
  inline const EphemerisData& GetEphemeris() const { return m_ephemeris; }
  inline bool EphemerisHasBeenSet() const { return m_ephemerisHasBeenSet; }
  template <typename EphemerisT = EphemerisData>
  void SetEphemeris(EphemerisT&& value) {
    m_ephemerisHasBeenSet = true;
    m_ephemeris = std::forward<EphemerisT>(value);
  }
  template <typename EphemerisT = EphemerisData>
  CreateEphemerisRequest& WithEphemeris(EphemerisT&& value) {
    SetEphemeris(std::forward<EphemerisT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags assigned to an ephemeris.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateEphemerisRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateEphemerisRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_satelliteId;

  bool m_enabled{false};

  int m_priority{0};

  Aws::Utils::DateTime m_expirationTime{};

  Aws::String m_name;

  Aws::String m_kmsKeyArn;

  EphemerisData m_ephemeris;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_satelliteIdHasBeenSet = false;
  bool m_enabledHasBeenSet = false;
  bool m_priorityHasBeenSet = false;
  bool m_expirationTimeHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_kmsKeyArnHasBeenSet = false;
  bool m_ephemerisHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace GroundStation
}  // namespace Aws
