﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/InstanceMetadataOptions.h>
#include <aws/imagebuilder/model/Logging.h>
#include <aws/imagebuilder/model/Placement.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class UpdateInfrastructureConfigurationRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API UpdateInfrastructureConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateInfrastructureConfiguration"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the infrastructure configuration that you
   * want to update.</p>
   */
  inline const Aws::String& GetInfrastructureConfigurationArn() const { return m_infrastructureConfigurationArn; }
  inline bool InfrastructureConfigurationArnHasBeenSet() const { return m_infrastructureConfigurationArnHasBeenSet; }
  template <typename InfrastructureConfigurationArnT = Aws::String>
  void SetInfrastructureConfigurationArn(InfrastructureConfigurationArnT&& value) {
    m_infrastructureConfigurationArnHasBeenSet = true;
    m_infrastructureConfigurationArn = std::forward<InfrastructureConfigurationArnT>(value);
  }
  template <typename InfrastructureConfigurationArnT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithInfrastructureConfigurationArn(InfrastructureConfigurationArnT&& value) {
    SetInfrastructureConfigurationArn(std::forward<InfrastructureConfigurationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the infrastructure configuration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance types of the infrastructure configuration. You can specify one
   * or more instance types to use for this build. The service will pick one of these
   * instance types based on availability.</p>
   */
  inline const Aws::Vector<Aws::String>& GetInstanceTypes() const { return m_instanceTypes; }
  inline bool InstanceTypesHasBeenSet() const { return m_instanceTypesHasBeenSet; }
  template <typename InstanceTypesT = Aws::Vector<Aws::String>>
  void SetInstanceTypes(InstanceTypesT&& value) {
    m_instanceTypesHasBeenSet = true;
    m_instanceTypes = std::forward<InstanceTypesT>(value);
  }
  template <typename InstanceTypesT = Aws::Vector<Aws::String>>
  UpdateInfrastructureConfigurationRequest& WithInstanceTypes(InstanceTypesT&& value) {
    SetInstanceTypes(std::forward<InstanceTypesT>(value));
    return *this;
  }
  template <typename InstanceTypesT = Aws::String>
  UpdateInfrastructureConfigurationRequest& AddInstanceTypes(InstanceTypesT&& value) {
    m_instanceTypesHasBeenSet = true;
    m_instanceTypes.emplace_back(std::forward<InstanceTypesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance profile to associate with the instance used to customize your
   * Amazon EC2 AMI.</p>
   */
  inline const Aws::String& GetInstanceProfileName() const { return m_instanceProfileName; }
  inline bool InstanceProfileNameHasBeenSet() const { return m_instanceProfileNameHasBeenSet; }
  template <typename InstanceProfileNameT = Aws::String>
  void SetInstanceProfileName(InstanceProfileNameT&& value) {
    m_instanceProfileNameHasBeenSet = true;
    m_instanceProfileName = std::forward<InstanceProfileNameT>(value);
  }
  template <typename InstanceProfileNameT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithInstanceProfileName(InstanceProfileNameT&& value) {
    SetInstanceProfileName(std::forward<InstanceProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The security group IDs to associate with the instance used to customize your
   * Amazon EC2 AMI.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  UpdateInfrastructureConfigurationRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  UpdateInfrastructureConfigurationRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subnet ID to place the instance used to customize your Amazon EC2 AMI
   * in.</p>
   */
  inline const Aws::String& GetSubnetId() const { return m_subnetId; }
  inline bool SubnetIdHasBeenSet() const { return m_subnetIdHasBeenSet; }
  template <typename SubnetIdT = Aws::String>
  void SetSubnetId(SubnetIdT&& value) {
    m_subnetIdHasBeenSet = true;
    m_subnetId = std::forward<SubnetIdT>(value);
  }
  template <typename SubnetIdT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithSubnetId(SubnetIdT&& value) {
    SetSubnetId(std::forward<SubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The logging configuration of the infrastructure configuration.</p>
   */
  inline const Logging& GetLogging() const { return m_logging; }
  inline bool LoggingHasBeenSet() const { return m_loggingHasBeenSet; }
  template <typename LoggingT = Logging>
  void SetLogging(LoggingT&& value) {
    m_loggingHasBeenSet = true;
    m_logging = std::forward<LoggingT>(value);
  }
  template <typename LoggingT = Logging>
  UpdateInfrastructureConfigurationRequest& WithLogging(LoggingT&& value) {
    SetLogging(std::forward<LoggingT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key pair of the infrastructure configuration. You can use this to log on
   * to and debug the instance used to create your image.</p>
   */
  inline const Aws::String& GetKeyPair() const { return m_keyPair; }
  inline bool KeyPairHasBeenSet() const { return m_keyPairHasBeenSet; }
  template <typename KeyPairT = Aws::String>
  void SetKeyPair(KeyPairT&& value) {
    m_keyPairHasBeenSet = true;
    m_keyPair = std::forward<KeyPairT>(value);
  }
  template <typename KeyPairT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithKeyPair(KeyPairT&& value) {
    SetKeyPair(std::forward<KeyPairT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The terminate instance on failure setting of the infrastructure
   * configuration. Set to false if you want Image Builder to retain the instance
   * used to configure your AMI if the build or test phase of your workflow
   * fails.</p>
   */
  inline bool GetTerminateInstanceOnFailure() const { return m_terminateInstanceOnFailure; }
  inline bool TerminateInstanceOnFailureHasBeenSet() const { return m_terminateInstanceOnFailureHasBeenSet; }
  inline void SetTerminateInstanceOnFailure(bool value) {
    m_terminateInstanceOnFailureHasBeenSet = true;
    m_terminateInstanceOnFailure = value;
  }
  inline UpdateInfrastructureConfigurationRequest& WithTerminateInstanceOnFailure(bool value) {
    SetTerminateInstanceOnFailure(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the SNS topic to which we send image build
   * event notifications.</p>  <p>EC2 Image Builder is unable to send
   * notifications to SNS topics that are encrypted using keys from other accounts.
   * The key that is used to encrypt the SNS topic must reside in the account that
   * the Image Builder service runs under.</p>
   */
  inline const Aws::String& GetSnsTopicArn() const { return m_snsTopicArn; }
  inline bool SnsTopicArnHasBeenSet() const { return m_snsTopicArnHasBeenSet; }
  template <typename SnsTopicArnT = Aws::String>
  void SetSnsTopicArn(SnsTopicArnT&& value) {
    m_snsTopicArnHasBeenSet = true;
    m_snsTopicArn = std::forward<SnsTopicArnT>(value);
  }
  template <typename SnsTopicArnT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithSnsTopicArn(SnsTopicArnT&& value) {
    SetSnsTopicArn(std::forward<SnsTopicArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags attached to the resource created by Image Builder.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetResourceTags() const { return m_resourceTags; }
  inline bool ResourceTagsHasBeenSet() const { return m_resourceTagsHasBeenSet; }
  template <typename ResourceTagsT = Aws::Map<Aws::String, Aws::String>>
  void SetResourceTags(ResourceTagsT&& value) {
    m_resourceTagsHasBeenSet = true;
    m_resourceTags = std::forward<ResourceTagsT>(value);
  }
  template <typename ResourceTagsT = Aws::Map<Aws::String, Aws::String>>
  UpdateInfrastructureConfigurationRequest& WithResourceTags(ResourceTagsT&& value) {
    SetResourceTags(std::forward<ResourceTagsT>(value));
    return *this;
  }
  template <typename ResourceTagsKeyT = Aws::String, typename ResourceTagsValueT = Aws::String>
  UpdateInfrastructureConfigurationRequest& AddResourceTags(ResourceTagsKeyT&& key, ResourceTagsValueT&& value) {
    m_resourceTagsHasBeenSet = true;
    m_resourceTags.emplace(std::forward<ResourceTagsKeyT>(key), std::forward<ResourceTagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance metadata options that you can set for the HTTP requests that
   * pipeline builds use to launch EC2 build and test instances. For more information
   * about instance metadata options, see one of the following links:</p> <ul> <li>
   * <p> <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/configuring-instance-metadata-options.html">Configure
   * the instance metadata options</a> in the <i> <i>Amazon EC2 User Guide</i> </i>
   * for Linux instances.</p> </li> <li> <p> <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/configuring-instance-metadata-options.html">Configure
   * the instance metadata options</a> in the <i> <i>Amazon EC2 Windows Guide</i>
   * </i> for Windows instances.</p> </li> </ul>
   */
  inline const InstanceMetadataOptions& GetInstanceMetadataOptions() const { return m_instanceMetadataOptions; }
  inline bool InstanceMetadataOptionsHasBeenSet() const { return m_instanceMetadataOptionsHasBeenSet; }
  template <typename InstanceMetadataOptionsT = InstanceMetadataOptions>
  void SetInstanceMetadataOptions(InstanceMetadataOptionsT&& value) {
    m_instanceMetadataOptionsHasBeenSet = true;
    m_instanceMetadataOptions = std::forward<InstanceMetadataOptionsT>(value);
  }
  template <typename InstanceMetadataOptionsT = InstanceMetadataOptions>
  UpdateInfrastructureConfigurationRequest& WithInstanceMetadataOptions(InstanceMetadataOptionsT&& value) {
    SetInstanceMetadataOptions(std::forward<InstanceMetadataOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance placement settings that define where the instances that are
   * launched from your image will run.</p>
   */
  inline const Placement& GetPlacement() const { return m_placement; }
  inline bool PlacementHasBeenSet() const { return m_placementHasBeenSet; }
  template <typename PlacementT = Placement>
  void SetPlacement(PlacementT&& value) {
    m_placementHasBeenSet = true;
    m_placement = std::forward<PlacementT>(value);
  }
  template <typename PlacementT = Placement>
  UpdateInfrastructureConfigurationRequest& WithPlacement(PlacementT&& value) {
    SetPlacement(std::forward<PlacementT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateInfrastructureConfigurationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_infrastructureConfigurationArn;

  Aws::String m_description;

  Aws::Vector<Aws::String> m_instanceTypes;

  Aws::String m_instanceProfileName;

  Aws::Vector<Aws::String> m_securityGroupIds;

  Aws::String m_subnetId;

  Logging m_logging;

  Aws::String m_keyPair;

  bool m_terminateInstanceOnFailure{false};

  Aws::String m_snsTopicArn;

  Aws::Map<Aws::String, Aws::String> m_resourceTags;

  InstanceMetadataOptions m_instanceMetadataOptions;

  Placement m_placement;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_infrastructureConfigurationArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_instanceTypesHasBeenSet = false;
  bool m_instanceProfileNameHasBeenSet = false;
  bool m_securityGroupIdsHasBeenSet = false;
  bool m_subnetIdHasBeenSet = false;
  bool m_loggingHasBeenSet = false;
  bool m_keyPairHasBeenSet = false;
  bool m_terminateInstanceOnFailureHasBeenSet = false;
  bool m_snsTopicArnHasBeenSet = false;
  bool m_resourceTagsHasBeenSet = false;
  bool m_instanceMetadataOptionsHasBeenSet = false;
  bool m_placementHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
