﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3CrtRequest.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/ChecksumAlgorithm.h>
#include <aws/s3-crt/model/ObjectLockLegalHold.h>
#include <aws/s3-crt/model/RequestPayer.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3Crt {
namespace Model {

/**
 */
class PutObjectLegalHoldRequest : public S3CrtRequest {
 public:
  AWS_S3CRT_API PutObjectLegalHoldRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutObjectLegalHold"; }

  AWS_S3CRT_API Aws::String SerializePayload() const override;

  AWS_S3CRT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3CRT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  AWS_S3CRT_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;
  AWS_S3CRT_API Aws::String GetChecksumAlgorithmName() const override;
  inline bool RequestChecksumRequired() const override { return true; };

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_S3CRT_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The bucket name containing the object that you want to place a legal hold on.
   * </p> <p> <b>Access points</b> - When you use this action with an access point
   * for general purpose buckets, you must provide the alias of the access point in
   * place of the bucket name or specify the access point ARN. When you use this
   * action with an access point for directory buckets, you must provide the access
   * point name in place of the bucket name. When using the access point ARN, you
   * must direct requests to the access point hostname. The access point hostname
   * takes the form
   * <i>AccessPointName</i>-<i>AccountId</i>.s3-accesspoint.<i>Region</i>.amazonaws.com.
   * When using this action with an access point through the Amazon Web Services
   * SDKs, you provide the access point ARN in place of the bucket name. For more
   * information about access point ARNs, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-access-points.html">Using
   * access points</a> in the <i>Amazon S3 User Guide</i>.</p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  PutObjectLegalHoldRequest& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key name for the object that you want to place a legal hold on.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  PutObjectLegalHoldRequest& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Container element for the legal hold configuration you want to apply to the
   * specified object.</p>
   */
  inline const ObjectLockLegalHold& GetLegalHold() const { return m_legalHold; }
  inline bool LegalHoldHasBeenSet() const { return m_legalHoldHasBeenSet; }
  template <typename LegalHoldT = ObjectLockLegalHold>
  void SetLegalHold(LegalHoldT&& value) {
    m_legalHoldHasBeenSet = true;
    m_legalHold = std::forward<LegalHoldT>(value);
  }
  template <typename LegalHoldT = ObjectLockLegalHold>
  PutObjectLegalHoldRequest& WithLegalHold(LegalHoldT&& value) {
    SetLegalHold(std::forward<LegalHoldT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline RequestPayer GetRequestPayer() const { return m_requestPayer; }
  inline bool RequestPayerHasBeenSet() const { return m_requestPayerHasBeenSet; }
  inline void SetRequestPayer(RequestPayer value) {
    m_requestPayerHasBeenSet = true;
    m_requestPayer = value;
  }
  inline PutObjectLegalHoldRequest& WithRequestPayer(RequestPayer value) {
    SetRequestPayer(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version ID of the object that you want to place a legal hold on.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  inline bool VersionIdHasBeenSet() const { return m_versionIdHasBeenSet; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  PutObjectLegalHoldRequest& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The MD5 hash for the request body.</p> <p>For requests made using the Amazon
   * Web Services Command Line Interface (CLI) or Amazon Web Services SDKs, this
   * field is calculated automatically.</p>
   */
  inline const Aws::String& GetContentMD5() const { return m_contentMD5; }
  inline bool ContentMD5HasBeenSet() const { return m_contentMD5HasBeenSet; }
  template <typename ContentMD5T = Aws::String>
  void SetContentMD5(ContentMD5T&& value) {
    m_contentMD5HasBeenSet = true;
    m_contentMD5 = std::forward<ContentMD5T>(value);
  }
  template <typename ContentMD5T = Aws::String>
  PutObjectLegalHoldRequest& WithContentMD5(ContentMD5T&& value) {
    SetContentMD5(std::forward<ContentMD5T>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the algorithm used to create the checksum for the object when you
   * use the SDK. This header will not provide any additional functionality if you
   * don't use the SDK. When you send this header, there must be a corresponding
   * <code>x-amz-checksum</code> or <code>x-amz-trailer</code> header sent.
   * Otherwise, Amazon S3 fails the request with the HTTP status code <code>400 Bad
   * Request</code>. For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/checking-object-integrity.html">Checking
   * object integrity</a> in the <i>Amazon S3 User Guide</i>.</p> <p>If you provide
   * an individual checksum, Amazon S3 ignores any provided
   * <code>ChecksumAlgorithm</code> parameter.</p>
   */
  inline ChecksumAlgorithm GetChecksumAlgorithm() const { return m_checksumAlgorithm; }
  inline bool ChecksumAlgorithmHasBeenSet() const { return m_checksumAlgorithmHasBeenSet; }
  inline void SetChecksumAlgorithm(ChecksumAlgorithm value) {
    m_checksumAlgorithmHasBeenSet = true;
    m_checksumAlgorithm = value;
  }
  inline PutObjectLegalHoldRequest& WithChecksumAlgorithm(ChecksumAlgorithm value) {
    SetChecksumAlgorithm(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account ID of the expected bucket owner. If the account ID that you
   * provide does not match the actual owner of the bucket, the request fails with
   * the HTTP status code <code>403 Forbidden</code> (access denied).</p>
   */
  inline const Aws::String& GetExpectedBucketOwner() const { return m_expectedBucketOwner; }
  inline bool ExpectedBucketOwnerHasBeenSet() const { return m_expectedBucketOwnerHasBeenSet; }
  template <typename ExpectedBucketOwnerT = Aws::String>
  void SetExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    m_expectedBucketOwnerHasBeenSet = true;
    m_expectedBucketOwner = std::forward<ExpectedBucketOwnerT>(value);
  }
  template <typename ExpectedBucketOwnerT = Aws::String>
  PutObjectLegalHoldRequest& WithExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    SetExpectedBucketOwner(std::forward<ExpectedBucketOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  PutObjectLegalHoldRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  PutObjectLegalHoldRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key, CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;

  Aws::String m_key;

  ObjectLockLegalHold m_legalHold;

  RequestPayer m_requestPayer{RequestPayer::NOT_SET};

  Aws::String m_versionId;

  Aws::String m_contentMD5;

  ChecksumAlgorithm m_checksumAlgorithm{ChecksumAlgorithm::NOT_SET};

  Aws::String m_expectedBucketOwner;

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_bucketHasBeenSet = false;
  bool m_keyHasBeenSet = false;
  bool m_legalHoldHasBeenSet = false;
  bool m_requestPayerHasBeenSet = false;
  bool m_versionIdHasBeenSet = false;
  bool m_contentMD5HasBeenSet = false;
  bool m_checksumAlgorithmHasBeenSet = false;
  bool m_expectedBucketOwnerHasBeenSet = false;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
