package Regexp::Common::zip;

use 5.10.0;

use strict;
use warnings;
no  warnings 'syntax';

use Regexp::Common qw /pattern clean no_defaults/;

our $VERSION = '2024080801';


#
# Prefer '[0-9]' over \d, because the latter may include more
# in Unicode string.
#

#
# ISO and Cept codes. ISO code is the second column, Cept code is
# the third. First column matches either.
#
# http://cept.org/ecc/topics/numbering-networks/numbering-related-
#        cooperation/the-cept-countries-joining-year-to-cept,
#        -cept-and-iso-country-codes,-e164-and-e212-country-codes
# (http://bit.ly/1Ue268b)
#
my %code = (
    Australia         =>  [qw /AUS? AU AUS/],
    Austria           =>  [qw /AU?T AT AUT/],
    Belgium           =>  [qw /BE?  BE B/],
    Denmark           =>  [qw /DK   DK DK/],
    France            =>  [qw /FR?  FR F/],
    Germany           =>  [qw /DE?  DE D/],
    Greenland         =>  [qw /GL   GL GL/],
    Italy             =>  [qw /IT?  IT I/],
    Liechtenstein     =>  [qw /LIE? LI LIE/],
    Luxembourg        =>  [qw /LU?  LU L/],
    Monaco            =>  [qw /MC   MC MC/],
    Netherlands       =>  [qw /NL   NL NL/],
    Norway            =>  [qw /NO?  NO N/],
   'San Marino'       =>  [qw /SM   SM SM/],
    Spain             =>  [qw /ES?  ES E/],
    Switzerland       =>  [qw /CH   CH CH/],
    USA               =>  [qw /USA? US USA/],
   'Vatican City'     =>  [qw /VA   VA VA/],
);

# Returns the empty string if the argument is undefined, the argument otherwise.
sub __ {defined $_ [0] ? $_ [0] : ""}

# Used for allowable options. If the value starts with 'y', the option is
# required ("{1,1}" is returned, if the value starts with 'n', the option
# is disallowed ("{0,0}" is returned), otherwise, the option is allowed,
# but not required ("{0,1}" is returned).
sub _t {
    if (defined $_ [0]) {
        if ($_ [0] =~ /^y/i) {return "{1,1}"}
        if ($_ [0] =~ /^n/i) {return "{0,0}"}
    }
    "{0,1}"
}

# Returns the (sub)pattern for the country named '$name', and the 
# -country option '$country'.
sub _c {
    my ($name, $country) = @_;
    if (defined $country && $country ne "") {
        if ($country eq 'iso')  {return $code {$name} [1]}
        if ($country eq 'cept') {return $code {$name} [2]}
        return $country;
    }
    $code {$name} [0]
}


my %zip = (
    #
    # Postal codes are four digits, but not all combinations are used.
    #
    # Valid codes from:
    #       https://en.wikipedia.org/wiki/List_of_postal_codes_in_Austria
    # 
    Austria =>
      "(?k:1(?:[0-8][0-9][0-9]|90[01])"                                      .
         "|2(?:[0-3][0-9][0-9]|"                                             .
              "4(?:0[0-9]|1[0-3]|2[1-5]|3[1-9]|[4-6][0-9]|7[0-5]|"           .
                  "8[1-9]|9[0-9])|"                                          .
              "[5-7][0-9][0-9]|"                                             .
              "8(?:[0-7][0-9]|8[01]))"                                       .
         "|3(?:0(?:0[1-9]|[1-9][0-9])|"                                      .
              "[12][0-9][0-9]|"                                              .
              "3(?:[0-2][0-9]|3[0-5]|[4-9][0-9])|"                           .
              "[4-8][0-9][0-9]|"                                             .
              "9(?:[0-6][0-9]|7[0-3]))"                                      .
         "|4(?:[01][0-9][0-9]|"                                              .
              "2(?:[0-8][0-9]|9[0-4])|"                                      .
              "3(?:0[0-3]|[1-8][0-9]|9[0-2])|"                               .
              "4(?:[0-1][0-9]|2[01]|3[1-9]|[4-9][0-9])|"                     .
              "[5-8][0-9][0-9]|"                                             .
              "9(?:[0-7][0-9]|8[0-5]))"                                      .
         "|5(?:0[0-9][0-9]|"                                                 .
              "1(?:0[0-9]|1[0-4]|[23][0-9]|4[0-5]|5[1-9]|[6-9][0-9])|"       .
              "2(?:0[0-5]|1[1-9]|[2-7][0-9]|8[0-3])|"                        .
              "3(?:0[0-3]|1[01]|2[1-9]|[34][0-9]|5[01]|60)|"                 .
              "[4-6][0-9][0-9]|"                                             .
              "7(?:[0-6][0-9]|7[01]))"                                       .
         "|6(?:[0-5][0-9][0-9]|"                                             .
              "6(?:[0-8][0-9]|9[01])|"                                       .
              "[78][0-9][0-9]|"                                              .
              "9(?:[0-8][0-9]|9[0-3]))"                                      .
         "|7(?:[0-3][0-9][0-9]|"                                             .
              "4(?:0[0-9]|1[0-3]|2[1-9]|[3-9][0-9])|"                        .
              "5(?:[0-6][0-9]|7[0-3]))"                                      .
         "|8(?:[0-2][0-9][0-9]|"                                             .
              "3(?:[0-5][0-9]|6[0-3]|8[0-5])|"                               .
              "4(?:0[1-9]|[1-9][0-9])|"                                      .
              "[5-8][0-9][0-9]|"                                             .
              "9(?:[0-8][0-9]|9[0-3]))"                                      .
         "|9(?:[0-6][0-9][0-9]|"                                             .
              "7(?:[0-7][0-9]|8[0-2])|"                                      .
              "8(?:[0-6][0-9]|7[0-3])|"                                      .
              "9(?:[0-8][0-9]|9[0-2]))"                                      .
    ")",
    #
    # Postal codes of the form: 'DDDD', with the first digit representing
    # the province; the others distribution sectors. Postal codes do not
    # start with a zero. Not all combinations are in use.
    #
    # Data from http://download.geonames.org/export/zip/BE.zip
    #
    Belgium  =>
      "(?k:1(?:0(?:0[05-9]|1[0-2]|20|3[01]|4[013-57-9]|50|60|70|8[0-3]|90)|" .
              "1(?:0[05]|10|20|30|40|50|60|70|80|90)|"                       .
              "2(?:0[01]|1[02])|"                                            .
              "3(?:0[01]|1[05]|2[05]|3[0-2]|4[0-28]|5[07]|6[07]|70|80|90)|"  .
              "4(?:0[0-24]|1[04]|2[018]|3[05]|40|5[07]|6[01]|"               .
                  "7[0-46]|80|9[05])|"                                       .
              "5(?:0[0-2]|4[017]|60|70)|"                                    .
              "6(?:0[0-2]|20|30|40|5[0-4]|7[0134])|"                         .
              "7(?:0[0-3]|3[01]|4[0-25]|5[05]|6[01]|70|8[05]|90)|"           .
              "8(?:0[04]|18|20|3[01]|40|5[0-3]|6[01]|80)|"                   .
              "9(?:10|3[0-4]|50|70|8[0-2]))"                                 .

         "|2(?:0(?:00|18|20|30|40|50|60|70)|"                                .
              "1(?:00|10|40|50|60|70|80)|"                                   .
              "2(?:00|2[0-3]|3[05]|4[023]|50|60|7[05]|8[08]|90)|"            .
              "3(?:00|10|2[0-38]|30|40|50|60|70|8[0-27]|90)|"                .
              "4(?:00|3[01]|40|50|60|70|80|9[01])|"                          .
              "5(?:00|20|3[01]|4[07]|50|60|70|80|90)|"                       .
              "6(?:00|10|2[07]|30|40|50|60)|"                                .
              "8(?:0[01]|1[12]|20|30|4[05]|50|6[01]|70|80|90)|"              .
              "9(?:00|10|20|30|40|50|60|70|80|90))"                          .

         "|3(?:0(?:0[01]|1[028]|20|40|5[0-4]|6[01]|7[018]|80|90)|"           .
              "1(?:1[018]|2[08]|30|40|50|9[01])|"                            .
              "2(?:0[0-2]|1[0-2]|2[01]|7[0-2]|9[034])|"                      .
              "3(?:00|2[01]|50|60|70|8[014]|9[01])|"                         .
              "4(?:0[014]|40|5[04]|6[01]|7[0-3])|"                           .
              "5(?:0[01]|1[0-2]|20|30|4[05]|50|60|70|8[0-3]|90)|"            .
              "6(?:00|2[01]|3[01]|40|50|6[058]|70|80|90)|"                   .
              "7(?:00|17|2[0-4]|3[02]|4[026]|70|9[0-38])|"                   .
              "8(?:0[036]|3[0-2]|40|50|70|9[01])|"                           .
              "9(?:00|10|20|30|4[015]|50|60|7[01]|80|90))"                   .

         "|4(?:0(?:00|20|3[0-2]|4[0-2]|5[0-3]|90)|"                          .
              "1(?:0[0-2]|2[0-2]|30|4[01]|51|6[0-3]|7[01]|8[01]|90)|"        .
              "2(?:1[07-9]|5[02-47]|6[013]|8[07])|"                          .
              "3(?:00|17|4[027]|5[017]|6[07])|"                              .
              "4(?:00|20|3[0-2]|5[0-38]|60|70|80)|"                          .
              "5(?:00|20|3[07]|40|5[07]|60|7[07]|90)|"                       .
              "6(?:0[0-26-8]|10|2[0134]|3[0-3]|5[0-4]|7[0-2]|8[0-4]|90)|"    .
              "7(?:0[01]|1[01]|2[018]|3[01]|50|6[01]|7[01]|8[02-4]|9[01])|"  .
              "8(?:0[0-2]|2[01]|3[0147]|4[015]|5[0-2]|6[01]|7[07]|80|90)|"   .
              "9(?:00|10|20|50|60|70|8[037]|90))" .

         "|5(?:0(?:0[0-4]|2[0-24]|3[0-2]|60|70|8[01])|"                      .
              "1(?:0[01]|40|50|70|90)|"                                      .
              "3(?:00|10|3[02-46]|40|5[0-4]|6[0-4]|7[02467]|80)|"            .
              "5(?:0[0-4]|2[0-4]|3[07]|4[0-4]|5[05]|6[0-4]|7[0-6]|80|90)|"   .
              "6(?:00|2[01]|30|4[0146]|5[01]|60|70|80))"                     .

         "|6(?:0(?:0[01]|10|20|3[0-2]|4[0-4]|6[01])|"                        .
              "1(?:1[01]|20|4[0-2]|50|8[0-3])|"                              .
              "2(?:00|1[01]|2[0-4]|3[08]|40|50|80)|"                         .
              "4(?:4[01]|6[0-4]|70)|"                                        .
              "5(?:00|11|3[0-46]|4[023]|6[07]|9[0-46])|"                     .
              "6(?:00|3[07]|4[02]|6[0-36]|7[0-4]|8[016-8]|9[028])|"          .
              "7(?:0[046]|17|2[0134]|30|4[0-37]|50|6[0-279]|8[0-2]|9[0-2])|" .
              "8(?:00|1[0-3]|2[0134]|3[0-468]|40|5[0-36]|60|70|8[07]|90)|"   .
              "9(?:00|2[0-2479]|4[01]|5[0-3]|60|7[0-2]|8[02-467]|9[07]))"    .

         "|7(?:0(?:00|1[0-2]|2[0-24]|3[0-4]|4[01]|50|6[0-3]|70|80|90)|"      .
              "1(?:00|10|20|3[0134]|4[01]|60|70|8[01]|9[01])|"               .
              "3(?:0[01]|2[0-2]|3[0-4]|40|50|70|8[027]|90)|"                 .
              "5(?:0[0-46]|2[0-2]|3[0-468]|4[0238])|"                        .
              "6(?:0[0-48]|1[018]|2[0-4]|4[0-3])|"                           .
              "7(?:00|1[12]|30|4[023]|50|60|8[0-4])|"                        .
              "8(?:0[0-4]|1[0-2]|2[23]|30|50|6[0-46]|70|80|90)|"             .
              "9(?:0[01346]|1[0-2]|4[0-3]|5[01]|7[0-3]))"                    .

         "|8(?:0(?:00|20)|"                                                  .
              "2(?:00|1[01])|"                                               .
              "3(?:0[01]|10|40|7[07]|80)|"                                   .
              "4(?:00|2[01]|3[0-4]|50|60|70|80|90)|"                         .
              "5(?:0[01]|1[01]|20|3[01]|40|5[0-4]|60|7[023]|8[0-37])|"       .
              "6(?:00|10|20|30|4[07]|50|60|70|80|9[01])|"                    .
              "7(?:00|10|20|30|40|5[05]|60|70|80|9[0-3])|"                   .
              "8(?:00|10|20|30|40|5[01]|60|70|80|90)|"                       .
              "9(?:0[02468]|20|30|40|5[0-46-8]|7[028]|80))"                  .

         "|9(?:0(?:00|3[0-2]|4[0-2]|5[0-2]|60|70|80|90)|"                    .
              "1(?:00|1[12]|20|30|40|50|60|70|8[05]|90)|"                    .
              "2(?:00|20|30|40|5[05]|60|70|80|90)|"                          .
              "3(?:0[08]|10|20|40)|"                                         .
              "4(?:0[0-46]|20|5[01]|7[023])|"                                .
              "5(?:0[06]|2[01]|5[0-2]|7[0-2])|"                              .
              "6(?:00|20|3[06]|6[017]|8[018]|90)|"                           .
              "7(?:00|50|7[0-2]|90)|"                                        .
              "8(?:00|10|20|3[01]|40|50|60|70|8[01]|90)|"                    .
              "9(?:00|10|2[01]|3[0-2]|40|50|6[018]|7[01]|8[0-28]|9[0-2]))"   .
    ")",

    #
    # Postal codes of the form: 'DDDD', with the first digit representing
    # the distribution region, the second digit the distribution district.
    # Postal codes do not start with a zero. Postal codes starting with '39'
    # are in Greenland, and not included in the pattern.
    #
    Denmark =>
      "(?k:0(?:800|"                                                         .
              "9(?:00|17|60|99))"                                            .

         "|1(?:0(?:00|5[0-9]|6[0-9]|7[0-4]|9[2358])|"                        .
              "1(?:0[0-7]|1[0-9]|2[0-9]|3[01]|4[078]|5[0-9]|6[0-24-9]|"      .
                  "7[0-5])|"                                                 .
              "2(?:0[0-9]|1[013-9]|2[01]|40|5[013-79]|6[013-8]|7[01]|91)|"   .
              "3(?:0[0-46-9]|1[0-9]|2[0-9]|5[02-9]|6[0-9]|7[01])|"           .
              "4(?:0[0-36-9]|1[0-9]|2[0-9]|3[0-9]|4[018]|5[0-9]|"            .
                  "6[02-46-8]|7[0-2])|"                                      .
              "5(?:00|13|3[23]|5[0-9]|6[0-46-9]|7[0-7]|9[29])|"              .
              "6(?:0[0-46-9]|1[0-9]|2[0-4]|3[0-5]|5[0-9]|6[0-9]|7[0-7]|99)|" .
              "7(?:0[0-9]|1[0-24-9]|2[0-9]|3[0-9]|49|5[0-9]|6[0-6]|"         .
                  "7[0-57]|8[05-7]|9[09])|"                                  .
              "8(?:0[0-9]|1[0-9]|2[02-9]|5[0-7]|6[0-8]|7[0-9])|"             .
              "9(?:0[0-689]|1[0-7]|2[0-8]|5[0-9]|6[0-7]|7[0-4]))"            .

         "|2(?:000|"                                                         .
              "1(?:00|50)|"                                                  .
              "200|"                                                         .
              "300|"                                                         .
              "4(?:00|50)|"                                                  .
              "500|"                                                         .
              "6(?:0[05]|10|2[05]|3[05]|40|50|6[05]|70|80|90)|"              .
              "7(?:00|20|30|40|50|6[05]|70|91)|"                             .
              "8(?:00|20|30|40|50|60|70|80)|"                                .
              "9(?:00|20|30|42|50|60|70|80|90))"                             .

         "|3(?:0(?:00|50|60|70|80)|"                                         .
              "1(?:00|20|40|50)|"                                            .
              "2(?:00|10|20|30|50)|"                                         .
              "3(?:00|10|20|30|60|70|90)|"                                   .
              "4(?:00|50|60|80|90)|"                                         .
              "5(?:00|20|40|50)|"                                            .
              "6(?:00|30|50|60|70)|"                                         .
              "7(?:00|20|30|40|51|60|70|82|90))"                             .

         "|4(?:0(?:00|30|40|50|60|70)|"                                      .
              "1(?:00|30|40|60|7[134]|80|90)|"                               .
              "2(?:00|20|30|4[1-3]|50|6[12]|70|81|9[1356])|"                 .
              "3(?:00|20|30|40|50|60|70|90)|"                                .
              "4(?:00|20|40|50|60|70|80|90)|"                                .
              "5(?:00|20|3[24]|40|50|60|7[1-3]|8[13]|9[1-3])|"               .
              "6(?:00|2[1-3]|32|40|5[2-4]|60|7[1-3]|8[1-4]|90)|"             .
              "7(?:00|20|3[356]|50|60|7[1-3]|80|9[1-3])|"                    .
              "8(?:00|40|50|6[23]|7[1-4]|80|9[1245])|"                       .
              "9(?:00|1[23]|20|30|4[134]|5[1-3]|60|70|83|90))"               .

         "|5(?:000|"                                                         .
              "2(?:00|10|20|30|40|50|60|70|90)|"                             .
              "3(?:00|20|30|50|70|80|90)|"                                   .
              "4(?:00|50|6[2-46]|7[14]|85|9[12])|"                           .
              "5(?:00|40|50|60|80|9[12])|"                                   .
              "6(?:00|10|20|31|42|72|83|90)|"                                .
              "7(?:00|50|62|7[12]|92)|"                                      .
              "8(?:00|5[346]|63|7[14]|8[1-4]|92)|"                           .
              "9(?:00|3[25]|53|60|70|85))"                                   .

         "|6(?:0(?:00|40|5[12]|64|70|9[1-4])|"                               .
              "100|"                                                         .
              "2(?:00|30|40|61|70|80)|"                                      .
              "3(?:00|10|20|30|40|60|72|92)|"                                .
              "4(?:00|30|40|70)|"                                            .
              "5(?:00|10|20|3[45]|41|60|80)|"                                .
              "6(?:00|2[1-3]|30|40|50|60|70|8[23]|90)|"                      .
              "7(?:0[05]|1[05]|20|31|40|5[23]|60|71|80|92)|"                 .
              "8(?:00|18|23|30|40|5[1-57]|62|70|80|93)|"                     .
              "9(?:00|20|33|40|50|60|7[13]|80|90))"                          .

         "|7(?:0(?:0[07]|80)|"                                               .
              "1(?:00|20|30|40|50|60|7[13]|8[2-4]|90)|"                      .
              "2(?:00|50|60|70|80)|"                                         .
              "3(?:00|2[13]|30|6[12])|"                                      .
              "4(?:00|30|4[12]|51|70|80|90)|"                                .
              "5(?:00|40|50|60|70)|"                                         .
              "6(?:00|20|50|60|73|80)|"                                      .
              "7(?:00|30|4[12]|5[25]|60|70|90)|"                             .
              "8(?:00|30|40|50|60|70|84)|"                                   .
              "9(?:00|50|60|70|80|90))"                                      .

         "|8(?:000|"                                                         .
              "2(?:00|10|20|30|4[05]|50|60|70)|"                             .
              "3(?:0[05]|10|20|30|40|5[05]|6[12]|70|8[0-2])|"                .
              "4(?:00|10|20|44|50|6[24]|7[12])|"                             .
              "5(?:00|20|30|4[134]|50|60|70|8[156]|92)|"                     .
              "6(?:00|20|32|4[13]|5[34]|60|70|80)|"                          .
              "7(?:00|2[1-3]|32|40|5[12]|6[2356]|8[13])|"                    .
              "8(?:00|3[0-2]|40|50|60|70|8[1-3])|"                           .
              "9(?:00|20|30|40|50|6[013]|70|8[13]|90))"                      .

         "|9(?:000|"                                                         .
              "2(?:00|10|20|30|40|60|70|80|93)|"                             .
              "3(?:00|10|20|30|40|52|62|70|8[0-2])|"                         .
              "4(?:00|30|40|60|80|9[023])|"                                  .
              "5(?:00|10|20|30|41|50|60|7[45])|"                             .
              "6(?:00|10|20|3[12]|40|70|81|90)|"                             .
              "7(?:00|40|50|60)|"                                            .
              "8(?:00|30|50|70|81)|"                                         .
              "9(?:00|40|70|8[12]|90))"                                      .
    ")",



    #
    # 5 Digit postal code, with leading 0s.
    #
    # Codes starting with 980 are reserved for Monaco, and not recognized
    # by the pattern.
    #
    # Data from: http://download.geonames.org/export/zip/FR.zip
    # 
    France      =>
      "(?k:0(?:1(?:0(?:0[0-9]|1[0-9]|20|59|6[019]|90)|1(?:0[0-9]|"           .
                  "1[0-25-7]|2[0-9]|30|40|5[0-58]|60|7[0-4]|90)|"            .
                  "2(?:0[0-468]|1[0-367]|20|30|40|50|60|70|80|90)|"          .
                  "3(?:0[0-36]|10|20|30|40|50|6[0569]|70|80|90)|4(?:00|"     .
                  "10|20|30|4[0-3]|50|60|70|80)|5(?:0[0-68]|10|40|50|60|"    .
                  "70|80|90)|6(?:0[0-46]|3[02378]|40|60|80)|7(?:0[0-8]|"     .
                  "10|50)|8(?:00|51)|9(?:21|60|90))|"                        .
              "2(?:0(?:0[0-9]|1[0-9]|2[05])|1(?:0[0-9]|10|20|30|40|5[01]|"   .
                  "60|70|90)|2(?:0[0-9]|10|20|30|40|50|60|70|90)|"           .
                  "3(?:0[0-39]|1[045]|2[0-79]|3[01]|40|50|60|70|80|90)|"     .
                  "4(?:0[0-79]|10|20|30|40|50|60|70|80|90)|5(?:00|10|20|"    .
                  "40|50|70|80|90)|6(?:0[0-49]|10|20|30|40|50|70|80|90)|"    .
                  "7(?:00|20|60|90)|8(?:00|10|20|30|40|50|60|70|80)|"        .
                  "93[09])|"                                                 .
              "3(?:0(?:0[03-8]|1[0-9]|21)|1(?:0[0-9]|10|20|30|40|50|60|"     .
                  "70|90)|2(?:0[0-9]|10|20|30|40|50|60|70|90)|"              .
                  "3(?:0[0-79]|1[01459]|2[019]|30|40|50|60|70|80|90)|"       .
                  "4(?:0[0-3569]|10|20|30|40|5[02]|60|70)|5(?:00|10)|"       .
                  "6(?:00|30)|700|800)|"                                     .
              "4(?:0(?:0[0-69]|1[0-9]|29)|1(?:0[0-7]|1[09]|20|30|40|50|"     .
                  "60|70|80|90)|2(?:0[0-39]|10|20|30|40|50|60|70|80|90)|"    .
                  "3(?:0[01]|10|20|30|40|50|60|70|80)|4(?:00|10|20)|"        .
                  "5(?:00|10|30)|6(?:00|60)|700|8(?:00|50|60|70)|99[05])|"   .
              "5(?:0(?:0[0-8]|1[0-24-69])|1(?:0[0-57]|10|20|30|40|50|60|"    .
                  "70|90)|2(?:0[0-289]|20|30|40|50|60|90)|3(?:00|10|20|"     .
                  "30|40|50|80)|4(?:00|60|70|80)|5(?:00|60)|600|700|800)|"   .
              "6(?:0(?:0[0-9]|1[0-36]|3[2-6]|4[4-9]|5[03]|7[1389]|8[2-5]|"   .
                  "99)|1(?:0[0-35-9]|1[03-79]|2[156]|3[0-35-79]|4[0-3]|"     .
                  "5[0-7]|6[0-247]|7[0-357]|8[0-79]|90)|2(?:0[0-69]|"        .
                  "1[0-2]|2[0-9]|3[0-9]|4[0-2]|5[0-579]|60|7[0-2]|"          .
                  "8[1-46-9]|9[02-59])|3(?:0[0-69]|10|20|30|4[0-589]|"       .
                  "5[2-47-9]|6[049]|7[0-369]|80|9[0-2])|4(?:0[0-8]|1[04]|"   .
                  "20|30|40|50|60|70|80)|5(?:0[0-8]|1[013-8]|20|3[0-245]|"   .
                  "40|50|60|70|80|90)|6(?:0[0-79]|10|20|3[1-4]|40|50|"       .
                  "6[06]|70|90)|7(?:0[0-689]|10|2[13]|30|40|50|90)|"         .
                  "8(?:0[0-68]|10|2[569]|3[013]|50|91)|9(?:0[1-69]|"         .
                  "1[02-5]|2[12589]|50|99))|"                                .
              "7(?:00[0-7]|1(?:0[0-9]|10|20|3[0-9]|40|50|60|70|90)|"         .
                  "2(?:0[0-9]|10|20|30|40|50|60|70|90)|3(?:0[0-9]|10|20|"    .
                  "30|40|50|60|70|80)|4(?:0[0-9]|10|30|40|50|60|70)|"        .
                  "5(?:0[0-9]|10|20|30|60|70|80|90)|6(?:00|10|30|60|90)|"    .
                  "7(?:00|90)|800)|"                                         .
              "8(?:0(?:0[0-6]|1[0134]|9[089])|1(?:0[1-79]|10|20|30|40|50|"   .
                  "60|70|90)|2(?:0[0-9]|10|20|30|40|50|60|70|90)|"           .
                  "3(?:0[02-5]|10|20|30|50|60|70|80|90)|4(?:00|10|30|40|"    .
                  "50|60)|500|600|700|800)|"                                 .
              "9(?:0(?:0[0-47-9]|1[4-7])|1(?:0[0-59]|10|20|30|40|60|90)|"    .
                  "2(?:0[019]|10|20|30|40|50|70|90)|3(?:0[01]|10|20|30|"     .
                  "40|50|90)|4(?:0[01]|20|60)|500|600|700|800))"             .

         "|1(?:0(?:0(?:0[0-9]|1[0-58]|2[56]|3[0-2]|42|8[0189]|9[126])|"      .
                  "1(?:0[0-5]|10|2[0-28]|30|40|5[0-4]|60|70|80|90)|"         .
                  "2(?:0[0-28]|10|20|30|40|50|60|70|80|90)|3(?:00|10|20|"    .
                  "30|40|5[0-35]|60|7[01]|80|90)|4(?:0[0-4]|10|20|3[0-3]|"   .
                  "40|50)|5(?:00|10)|60[0-356]|700|800|9(?:0[12]|10))|"      .
              "1(?:0(?:0[0-579]|1[0-25-7]|2[0-29])|1(?:0[0-9]|10|20|30|"     .
                  "40|50|60|70|90)|2(?:0[0-589]|10|20|30|40|50|60|70|90)|"   .
                  "3(?:0[0134]|10|20|30|40|50|60|70|80|90)|4(?:0[0-29]|"     .
                  "10|20|30|40|5[1-3]|80|9[0-4])|5(?:00|10|40|60|70|80|"     .
                  "90)|6(?:00|10|20)|7(?:00|8[12459]|90)|8(?:0[0235-8]|"     .
                  "1[06]|23|3[3568]|48|5[05]|60|7[05-8]|8[05]|9[08]))|"      .
              "2(?:0(?:0[0-357-9]|19|2[0-9]|3[0-59]|40)|1(?:0[0-4689]|10|"   .
                  "20|30|40|50|60|70|90)|2(?:0[0-4]|10|20|30|40|50|60|70|"   .
                  "90)|3(?:00|10|20|30|40|50|60|70|80|90)|4(?:0[0-2]|10|"    .
                  "20|30|40|50|60|70|80|90)|5(?:00|10|20|40|50|60|80)|"      .
                  "6(?:00|20|30|40)|7(?:00|20|40|80)|8(?:00|50))|"           .
              "3(?:0(?:0[0-9]|1[0-6]|20|6[67]|8[0589]|9[0-9])|1(?:0[0-9]|"   .
                  "1[0-9]|2[0-46-9]|3[0-389]|4[0-289]|5[0-35-9]|"            .
                  "6[013-589]|7[078]|8[0-2]|9[0-26])|2(?:0[0-3579]|"         .
                  "1[013-9]|2[0-8]|3[0-69]|4[0-589]|5[0-9]|6[0-9]|7[0-9]|"   .
                  "8[0-24-9]|9[0-9])|3(?:0[0-46-9]|1[0-9]|2[0-8]|"           .
                  "3[0-589]|4[02-8]|5[0-24-79]|6[0-46-9]|7[0-24-9]|"         .
                  "8[0-9]|9[0-79])|4(?:0[0-46]|1[0-7]|2[014-68]|3[0-245]|"   .
                  "4[0-8]|5[0-8]|6[02-47]|7[0-48]|8[0-49]|9[03])|"           .
                  "5(?:0[0-3]|1[0-4689]|2[0-589]|3[0-389]|4[0-35-9]|"        .
                  "5[0-289]|6[06-8]|7[02]|8[013]|9[0-49])|6(?:0[0-9]|"       .
                  "1[0-24-79]|2[015-9]|3[0-357]|4[0-369]|5[0-689]|6[0-9]|"   .
                  "7[0-8]|8[013-57-9]|9[0-9])|7(?:0[0-589]|1[02-9]|"         .
                  "2[0-9]|3[0-3]|4[0-79]|5[0-257-9]|6[046]|7[0-9]|"          .
                  "8[0-59]|9[0-9])|8(?:0[0-9]|1[02-5]|2[0-589]|3[0-689]|"    .
                  "4[0-24-7]|5[0-9]|6[0-478]|70|8[01379]|9[05])|"            .
                  "9(?:0[024-8]|1[01]|2[0-4]|3[0-37]|4[0-2]|50|60|80|90))|"  .
              "4(?:0(?:0[05-9]|1[0-9]|2[03-9]|3[0-9]|4[05-8]|5[0-489]|"      .
                  "6[1-7]|7[04-9]|8[1-69]|9[0-9])|1(?:0[0-9]|1[0-47]|"       .
                  "2[0135-8]|30|40|50|6[0-258]|70|90)|2(?:0[0-57-9]|10|"     .
                  "20|30|40|50|60|70|80|90)|3(?:10|20|30|40|50|60|70|80|"    .
                  "90)|4(?:0[0-9]|10|20|30|40|50|6[0-29]|70|80|90)|"         .
                  "5(?:0[0-589]|10|20|30|40|50|70|90)|6(?:0[0-3]|10|20|"     .
                  "30|40|5[0-49]|70|80|90)|7(?:00|10|30|40|50|60|70|80|"     .
                  "9[01])|8(?:0[0-9]|10|30|40|50|60|80)|9(?:0[1-9]|"         .
                  "1[0-69]|2[0-6]|3[0-4]|4[09]|50|60|70|80|90))|"            .
              "5(?:0(?:0[0-8]|1[0-57-9])|1(?:0[0-79]|10|20|30|40|50|60|"     .
                  "70|90)|2(?:00|10|2[09]|30|40|50|60|70|90)|3(?:00|10|"     .
                  "20|40|50|80)|4(?:00|30)|5(?:00|90)|600|700|800)|"         .
              "6(?:0(?:0[0-8]|1[0-9]|2[0-6])|1(?:0[0-9]|1[0-3]|2[01]|30|"    .
                  "40|50|60|70|90)|2(?:00|10|20|30|40|50|60|70|90)|"         .
                  "3(?:00|10|20|30|40|50|60|70|80|90)|4(?:00|10|20|30|40|"   .
                  "50|60|70|80|90)|5(?:00|10|60|70|90)|6(?:00|20)|7(?:00|"   .
                  "10|20|30)|800|9(?:0[1-9]|1[0-9]|2[09]|5[0-59]|99))|"      .
              "7(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-5]|5[1-6]|"           .
                  "7[1-689]|8[6-9])|1(?:0[0-9]|1[0-9]|2[013]|3[0236-9]|"     .
                  "40|50|60|70|8[0-9]|90)|2(?:0[0-9]|1[0145]|20|30|40|50|"   .
                  "60|70|8[1459]|90)|3(?:0[0-9]|1[0-4]|20|30|40|50|60|70|"   .
                  "80|90)|4(?:00|1[0-69]|20|30|4[0-69]|50|60|70|80|90)|"     .
                  "5(?:0[0-49]|10|20|30|40|50|60|70|80|90)|6(?:00|10|20|"    .
                  "30|40|50|70|90)|7(?:00|30|40|50|70|80)|8(?:00|10|40|"     .
                  "70|80|90)|9(?:20|40))|"                                   .
              "8(?:0(?:0[0-7]|1[2-69]|2[0-9]|3[0-79])|1(?:0[0-9]|10|20|"     .
                  "30|40|50|60|70|90)|2(?:0[0-7]|10|20|30|40|50|60|70|"      .
                  "90)|3(?:00|10|20|30|40|50|60|70|80|90)|4(?:00|10)|"       .
                  "5(?:00|10|20|70)|600|700|800|9(?:1[01]|2[02-4]|3[459]|"   .
                  "4[015]|98))|"                                             .
              "9(?:0(?:0[0-79]|1[1257-9]|33)|1(?:0[0-9]|1[01346-9]|20|30|"   .
                  "40|50|60|70|90)|2(?:0[0-489]|10|20|3[0-2]|40|50|60|70|"   .
                  "90)|3(?:00|1[0-9]|20|30|40|50|6[01]|70|80|90)|4(?:00|"    .
                  "10|30|50|60|70|90)|5(?:00|10|20|50|60)|600|700|800))"     .

         "|2(?:0(?:0(?:00|90)|1(?:00|1[0-9]|2[1-9]|3[0-9]|4[0-8]|5[0-37]|"   .
                  "6[02-9]|7[0-9]|8[0-9]|9[0-57-9])|2(?:00|1[2-57-9]|"       .
                  "2[0-9]|3[0-9]|4[02-8]|5[0-369]|60|7[02569]|8[7-9]|"       .
                  "9[0-9])|3(?:0[2-69]|11)|4(?:0[1-35-9]|1[0-46-9])|"        .
                  "5(?:0[1-4]|3[78])|6(?:0[014]|11|20)|70[0-3]|900)|"        .
              "1(?:0(?:0[0-9]|1[0-9]|2[1-9]|3[0-9]|4[0-9]|5[1-9]|"           .
                  "6[0-35-9]|7[0-9]|8[013-689]|9[2378])|1(?:10|2[01]|30|"    .
                  "40|50|60|70|90)|2(?:0[0-9]|1[09]|20|30|40|50|6[0-2]|"     .
                  "70|90)|3(?:0[0-59]|10|20|30|40|50|60|70|80|90)|"          .
                  "4(?:0[0-39]|10|20|30|40|50|60|70|90)|5(?:0[0-269]|10|"    .
                  "20|30|40|50|60|70|80|90)|6(?:0[0-49]|10|30|40|90)|"       .
                  "7(?:0[0-59]|19|60)|8(?:0[0-369]|20|50)|9(?:00|10|98))|"   .
              "2(?:0(?:0[0-5]|1[4-7]|2[1-79]|3[1-5]|4[1-69]|7[09]|8[09]|"    .
                  "9[1-35689])|1(?:0[0-9]|1[01]|20|30|40|50|60|70|"          .
                  "9[0-69])|2(?:0[0-69]|10|20|30|40|50|60|70|90)|"           .
                  "3(?:0[0-9]|10|20|30|40|50|60|70|80|90)|4(?:0[02-59]|"     .
                  "10|20|30|40|50|60|70|80|90)|5(?:0[0-59]|10|20|30|40|"     .
                  "50|60|70|80|90)|6(?:0[02-79]|10|20|30|40|50|60|80|90)|"   .
                  "7(?:00|10|20|30|40|50|70|80)|8(?:00|10|20|30|60|70)|"     .
                  "9(?:30|40|50|60|70|80))|"                                 .
              "3(?:0(?:0[0-9]|1[13-6]|20)|1(?:00|10|20|30|40|50|60|70|"      .
                  "90)|2(?:00|10|20|30|40|50|60|70|90)|3(?:00|20|40|50|"     .
                  "60|80)|4(?:00|20|30|50|60|80)|500|600|700|800)|"          .
              "4(?:0(?:0[0-579]|1[0-79]|2[0249]|5[0-359]|60)|"               .
                  "1(?:0[0-24-9]|1[0-4]|2[0-2]|30|40|50|60|70|90)|"          .
                  "2(?:0[0-689]|1[02]|20|30|40|50|60|70|90)|3(?:00|10|20|"   .
                  "30|40|50|60|70|80|90)|4(?:00|10|20|30|40|50|60|70|80|"    .
                  "90)|5(?:00|10|20|30|40|50|60|70|80|90)|6(?:00|10|20|"     .
                  "30|40|50|60|80)|7(?:00|5[0-2589])|800|9(?:1[0235]|26))|"  .
              "5(?:0(?:0[0-9]|1[0-9]|2[0-79]|3[0-9]|4[0-9]|5[0-246-9]|"      .
                  "6[0-389]|7[0-359]|8[02-79]|9[089])|1(?:1[0-9]|20|30|"     .
                  "40|50|60|70|90)|2(?:0[0-9]|1[0-9]|20|30|40|50|60|70|"     .
                  "90)|3(?:0[0-49]|10|20|30|40|50|60|70|80|90)|"             .
                  "4(?:0[0-69]|10|20|30|40|50|6[0-29]|7[09]|80|90)|"         .
                  "5(?:0[0-489]|10|20|30|50|60|70|80)|6(?:0[0-9]|10|20|"     .
                  "30|40|50|60|80|90)|7(?:0[0-26-9]|20|50|70|90)|8(?:00|"    .
                  "20|40|70)|9(?:09|20|3[0-3]|60))|"                         .
              "6(?:0(?:0[0-9]|1[0-5]|2[1-9]|3[0-2])|1(?:0[0-9]|1[019]|20|"   .
                  "3[01]|40|50|60|70|9[09])|2(?:0[0-9]|1[056]|20|30|"        .
                  "4[0-49]|50|60|70|90)|3(?:0[0-49]|10|20|30|40|50|80|"      .
                  "90)|4(?:0[0-29]|10|20|50|60|70)|5(?:0[0-49]|10|30|40|"    .
                  "60|70)|6(?:0[0-39]|20)|7(?:0[0-29]|30|40|50|6[01]|70|"    .
                  "80|90)|80[0-29]|9(?:0[1-79]|5[0-68]))|"                   .
              "7(?:0(?:0[0-9]|1[0235-9]|2[1-356]|3[0-9]|40|9[0-9])|"         .
                  "1(?:0[0-9]|10|2[0-27]|3[0-8]|40|50|60|70|80|90)|"         .
                  "2(?:0[0-9]|10|20|30|40|50|60|70|90)|3(?:0[0-9]|10|20|"    .
                  "30|40|50|60|70|80|90)|4(?:0[0-79]|10|20|30|40|50|60|"     .
                  "70|80|90)|5(?:0[0-9]|10|20|30|40|50|60|70|80|90)|"        .
                  "6(?:0[07]|10|20|30|40|50|60|70|80|90)|7(?:0[0-35]|10|"    .
                  "20|30|40|50|60|70|80|90)|8(?:00|10|20|30|50|60|70|90)|"   .
                  "9(?:09|1[014]|2[0489]|3[0-369]|4[09]|50))|"               .
              "8(?:0(?:0[0-9]|1[189]|2[3-69]|3[3-9]|4[04]|8[08])|"           .
                  "1(?:0[0-689]|1[0-49]|20|3[0-3]|40|50|60|70|90)|"          .
                  "2(?:0[0-3579]|1[01]|20|3[0-24]|40|50|60|70|90)|"          .
                  "3(?:0[0-59]|10|20|30|40|50|60|80)|4(?:0[0-49]|10|"        .
                  "8[019])|50[01]|6(?:00|3[0-3679])|70[0-379]|800|9(?:01|"   .
                  "10|20|3[0-35]|44|50))|"                                   .
              "9(?:0(?:00|18|80)|1(?:0[0-9]|2[0-59]|40|50|60|7[0-4679]|"     .
                  "8[0-79]|9[0-369])|2(?:0[06-9]|1[0-37-9]|2[0-57-9]|"       .
                  "3[138]|4[0-26]|5[0239]|60|70|8[029]|9[09])|3(?:00|10|"    .
                  "2[0-8]|3[0-7]|40|50|60|70|80|9[0-9])|4(?:0[0-4679]|"      .
                  "1[0-49]|20|30|40|5[05]|60|70|80|90)|5(?:00|10|20|30|"     .
                  "40|5[0-9]|6[013]|70|90)|6(?:0[0269]|10|20|30|40|50|60|"   .
                  "7[0-489]|8[0-2489]|90)|7(?:00|10|20|30|40|50|60|70|80|"   .
                  "90)|8(?:0[0-46-9]|10|2[0457]|3[03-79]|40|50|60|70|80|"    .
                  "90)|9(?:00|10|20|30|40|50|70|80|90)))"                    .

         "|3(?:0(?:0(?:0[0-36-9]|1[0-9]|2[0-589]|3[1-69]|4[04589]|"          .
                  "5[015])|1(?:0[0-79]|1[0-2459]|2[0-9]|3[0-49]|40|50|60|"   .
                  "70|90)|2(?:0[0-79]|10|20|30|40|5[0-35]|60|70|90)|"        .
                  "3(?:0[0-2]|1[01389]|20|30|40|50|60|80|90)|4(?:0[0-69]|"   .
                  "10|20|30|40|50|60|70|80|90)|5(?:00|10|20|30|40|60|70|"    .
                  "80)|6(?:00|10|20|30|40|50|60|70)|7(?:0[0-3]|20|30|40|"    .
                  "50|60|70)|8(?:00|20|40|70)|9(?:0[013-8]|1[0-48]|"         .
                  "2[0-59]|3[124-79]|4[0-37]|6[09]|7[125]|80))|"             .
              "1(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|"    .
                  "7[0-9]|8[0-24-689]|9[0-9])|1(?:0[0-4679]|1[02]|2[0-9]|"   .
                  "3[0-9]|4[0-29]|5[0-259]|60|70|8[09]|90)|2(?:0[013-5]|"    .
                  "10|20|30|4[0-59]|50|60|70|8[09]|90)|3(?:00|1[02-579]|"    .
                  "2[0-2569]|30|40|50|60|70|8[09]|90)|4(?:0[0-6]|10|20|"     .
                  "3[02]|40|50|60|70|8[09]|90)|5(?:0[03-7]|1[02]|"           .
                  "2[0-249]|30|40|50|60|70|8[09]|90)|6(?:0[0-689]|2[019]|"   .
                  "50|60|7[0-9]|8[1-359]|92)|7(?:0[0-9]|1[1256]|50|"         .
                  "7[0-4679]|80|90)|8(?:0[0-79]|10|20|3[0-29]|40|50|60|"     .
                  "70|80)|9(?:0[0-3]|31|4[57]|5[078]|6[02]|9[89]))|"         .
              "2(?:0(?:0[0-57-9]|1[0-46-9]|2[0-2])|1(?:00|10|2[09]|30|40|"   .
                  "50|60|70|90)|2(?:0[019]|20|30|40|50|60|70|90)|3(?:00|"    .
                  "10|20|30|40|50|60|70|80|90)|4(?:00|10|20|30|40|50|60|"    .
                  "80|90)|5(?:0[0-259]|50)|600|7(?:00|20|30)|8(?:00|10))|"   .
              "3(?:0(?:0[0-24-9]|1[0-259]|2[013-9]|3[0-8]|4[0-37-9]|"        .
                  "5[0-9]|6[0-9]|7[0-9]|8[0-9]|9[0-9])|1(?:00|1[02-6]|"      .
                  "2[013-7]|3[038]|4[018]|5[0-35]|6[04-79]|7[0-5]|"          .
                  "8[05-7]|9[0-3])|2(?:00|1[0-49]|20|30|40|50|60|"           .
                  "7[0-489]|9[0-5])|3(?:0[0569]|1[0-49]|2[0-4679]|3[05]|"    .
                  "4[01]|50|60|70|80|9[0-5])|4(?:0[0-59]|10|20|30|40|"       .
                  "5[0-2]|6[019]|70|80|9[0-25])|5(?:0[0-79]|1[05]|"          .
                  "2[0-79]|30|40|50|6[0-5]|70|80|90)|6(?:0[0-9]|1[0-5]|"     .
                  "2[015-79]|40|5[02]|60|70|8[089]|9[0-9])|7(?:0[0-9]|10|"   .
                  "20|3[0-9]|4[017]|50|60|70|80|90)|8(?:0[01]|10|20|30|"     .
                  "40|50|60|70|8[0-7]|90)|9(?:0[0-79]|1[0-589]|20|30|50|"    .
                  "70|80|9[08]))|"                                           .
              "4(?:0(?:0[0-46-9]|1[0-2]|2[3-8]|3[02-57-9]|4[0-35689]|"       .
                  "5[13-68]|6[0-8]|7[0-9]|8[0235-79]|9[0-9])|1(?:1[0-9]|"    .
                  "20|3[0-27-9]|40|50|60|7[0-489]|8[13467]|9[0-9])|"         .
                  "2(?:0[0-479]|10|20|30|40|50|6[0-7]|7[04]|80|9[03-9])|"    .
                  "3(?:0[0-579]|10|20|30|40|50|60|70|80|9[0-46-9])|"         .
                  "4(?:0[0-49]|10|20|3[013-589]|40|50|60|7[037]|80|90)|"     .
                  "5(?:0[0-8]|1[03-8]|2[0-69]|3[04-79]|4[03-5]|50|6[04]|"    .
                  "70|90)|6(?:00|10|20|30|50|60|7[01]|80|90)|7(?:0[0-2]|"    .
                  "10|2[05]|30|4[0-27-9]|5[013]|60|70|90)|8(?:00|10|20|"     .
                  "30|50|7[1-5]|80)|9(?:00|2[0356]|3[2-57-9]|4[02-489]|"     .
                  "5[4-9]|6[0-2457]|7[02-57]|8[0-79]|90))|"                  .
              "5(?:0(?:0[0-9]|1[0-24-68]|2[0-46-9]|3[0-59]|4[02-79]|"        .
                  "5[0-57-9]|6[3-79]|7[013689]|8[0-9]|9[0-49])|"             .
                  "1(?:0[1-689]|1[134]|20|3[0-7]|40|50|6[0-469]|"            .
                  "7[0-24-7]|90)|2(?:0[0-57-9]|10|2[01]|3[056]|40|50|60|"    .
                  "70|90)|3(?:0[0-69]|10|20|30|4[0-259]|50|60|70|80|90)|"    .
                  "4(?:0[0-9]|1[0-9]|20|3[0-359]|40|50|60|70|80|90)|"        .
                  "5(?:0[0-79]|1[0-9]|2[01]|3[0-389]|40|50|60|7[1-46-9]|"    .
                  "80|90)|6(?:0[0-356]|10|20|30|40|5[0-39]|60|80|90)|"       .
                  "7(?:0[0-9]|11|20|30|4[0-46]|50|6[0-9]|7[0-2]|80)|"        .
                  "8(?:0[0-3]|3[0-3]|50|70|90)|9(?:0[0-9]|1[1-9]|2[01]|"     .
                  "60|98))|"                                                 .
              "6(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[23])|1(?:0[0-579]|10|20|"      .
                  "3[01]|40|50|60|70|80|90)|2(?:00|10|20|30|40|5[05]|60|"    .
                  "70|90)|3(?:0[01]|10|20|30|40|50|60|70)|400|50[015]|"      .
                  "600|700|800|91[05])|"                                     .
              "7(?:0(?:0[089]|1[0-9]|2[05-9]|3[2-589]|4[0-57-9]|5[89]|"      .
                  "6[01]|7[1-9]|8[0-29]|9[5-9])|1(?:00|10|20|30|40|5[02]|"   .
                  "60|7[0-59]|90)|2(?:0[04-69]|10|20|30|40|50|60|70|90)|"    .
                  "3(?:0[0-69]|10|2[01]|30|40|50|60|70|80|90)|"              .
                  "4(?:0[0-49]|20|60)|5(?:0[0-29]|10|2[01]|30|4[0-29]|"      .
                  "5[0-6])|60[0-29]|70[0-359]|800|9(?:1[0-357]|2[124-9]|"    .
                  "3[12]|4[12]))|"                                           .
              "8(?:0(?:0[0-9]|1[0-9]|2[1-9]|3[0-79]|4[0-9]|5[0-4689]|"       .
                  "6[135-79]|7[047]|8[019]|9[0-36-9])|1(?:00|1[02-489]|"     .
                  "2[0-2]|3[048]|4[0246-8]|5[02-49]|6[0-59]|7[0-469]|80|"    .
                  "9[0167])|2(?:0[0-9]|1[016-9]|20|3[0-26]|4[0-469]|"        .
                  "5[06]|6[019]|70|80|9[0-35-79])|3(?:0[0-579]|1[1-579]|"    .
                  "2[0-2679]|3[0-4]|4[0-35-79]|5[0-46-9]|6[01]|70|80|90)|"   .
                  "4(?:0[0-9]|10|20|3[0-79]|40|50|60|70|80|90)|"             .
                  "5(?:0[0-9]|1[0169]|2[0-469]|30|40|5[0-79]|60|70|80|"      .
                  "9[01])|6(?:0[0-39]|10|20|30|40|50|60|70|80|90)|"          .
                  "7(?:0[0-26-9]|10|3[01]|40|50|6[01379]|70|8[0-249]|90)|"   .
                  "8(?:0[0-3]|1[67]|21|30|40|50|60|70|80|90)|9(?:00|13|"     .
                  "2[0167]|30|4[0134]|50|60|70|80))|"                        .
              "9(?:0(?:0[0-9]|1[056]|2[19]|3[0-9])|1(?:0[0-9]|10|20|30|"     .
                  "40|50|60|7[01]|8[019]|90)|2(?:0[0-9]|1[019]|20|30|40|"    .
                  "50|6[01]|70|90)|3(?:0[0-9]|10|20|30|50|60|70|80)|"        .
                  "4(?:0[0-9]|10|60)|5(?:0[0-2]|20|70)|60[0-359]|700|"       .
                  "80[0-9]))"                                                .

         "|4(?:0(?:0(?:0[0-6]|1[1-35-9]|2[1-7]|90)|1(?:0[0-8]|1[058]|20|"    .
                  "30|4[01]|50|6[01]|70|80|90)|2(?:0[0-2]|10|20|3[0-259]|"   .
                  "40|50|60|70|8[0-26]|90)|3(?:0[0159]|10|20|30|50|60|70|"   .
                  "80|90)|4(?:00|10|20|30|40|6[05]|80)|5(?:0[0-2]|10|30|"    .
                  "50|60)|6(?:0[0-2]|30|60)|70[015]|80[015]|99[0-4])|"       .
              "1(?:0(?:0[0-9]|1[0-35689]|2[0235689]|3[3-5]|4[23])|"          .
                  "1(?:0[0-3569]|10|2[0-2]|30|40|5[01]|60|70|90)|"           .
                  "2(?:0[0-7]|10|20|30|40|50|6[01]|70|90)|3(?:00|10|20|"     .
                  "30|5[03-57]|60|70)|40[0-26]|500|600|700|800|"             .
                  "9(?:0[13-689]|1[03-589]|2[145]|3[0-36]|4[1-3589]|"        .
                  "5[018]|6[03-8]|7[0-6]))|"                                 .
              "2(?:0(?:0[0-9]|1[0-8]|2[1-9]|3[013]|4[1-357-9]|5[0589])|"     .
                  "1(?:00|1[014]|2[02-79]|3[01]|40|5[0235]|6[0-9]|"          .
                  "7[03469]|90)|2(?:10|20|30|40|60|7[0-35-79]|90)|"          .
                  "3(?:0[0-48]|1[0-58]|2[0-38]|3[02-59]|40|5[013-69]|60|"    .
                  "70|80|90)|4(?:0[0-9]|1[0-2]|20|30|40|50|60|70|8[0469]|"   .
                  "90)|5(?:0[0-359]|10|20|30|40|50|60|70|80|90)|"            .
                  "6(?:0[0-9]|10|20|30|40|5[013]|60|70|80)|7(?:0[0-579]|"    .
                  "20|40|50|80)|8(?:0[0-9]|10|20|30|40|90)|9(?:20|40|"       .
                  "5[0-589]|6[1-9]|90))|"                                    .
              "3(?:0(?:0[0-689]|1[0-479])|1(?:0[0-3]|10|20|30|40|50|60|"     .
                  "70|90)|2(?:0[0-3]|10|20|30|40|50|60|70|90)|3(?:00|20|"    .
                  "30|40|50|60|70|80|90)|4(?:00|10|20|30|40|50|90)|"         .
                  "5(?:00|10|20|30|50|80|90)|6(?:00|20)|7(?:00|50|70)|"      .
                  "8(?:00|10))|"                                             .
              "4(?:0(?:0[0-9]|1[0-9]|2[0-4]|3[2-68]|4[0-267]|9[02-579])|"    .
                  "1(?:0[0-79]|1[05-9]|2[0-469]|30|4[0-69]|5[0-9]|60|"       .
                  "7[069]|8[4-8]|9[0-689])|2(?:0[0-59]|1[0-79]|20|"          .
                  "3[0-69]|4[0-59]|50|6[0-358]|7[02-7]|90)|3(?:0[0-9]|"      .
                  "1[0-9]|2[0-9]|3[0-9]|4[0-69]|5[0-36-9]|60|7[0-39]|80|"    .
                  "90)|4(?:0[0-9]|1[02]|20|30|40|50|60|7[0-9]|8[0-4]|90)|"   .
                  "5(?:0[0-69]|10|2[0-2]|30|40|50|60|70|80|90)|"             .
                  "6(?:0[0-69]|1[0-8]|20|30|40|50|60|70|80|90)|"             .
                  "7(?:0[0-36-9]|10|20|30|40|50|60|70|80)|8(?:0[0-9]|"       .
                  "1[0-9]|2[1-49]|30|40|50|60|80)|9(?:00|1[1-9]|2[1-9]|"     .
                  "3[1-9]|4[124579]|5[1-9]|6[3-9]|7[1-8]|8[0-69]))|"         .
              "5(?:0(?:0[0-9]|1[0-25-79]|2[13589]|3[0-58]|4[0-6]|"           .
                  "5[236-8]|6[0-57-9]|7[1-57-9]|8[1278])|1(?:00|10|"         .
                  "2[0-259]|30|4[0-79]|50|6[0-689]|70|90)|2(?:0[0-9]|"       .
                  "1[0-4]|20|30|40|50|60|70|90)|3(?:0[02-8]|1[02]|20|"       .
                  "3[01]|40|60|70|80|90)|4(?:0[0-469]|10|20|3[0-3]|50|60|"   .
                  "70|80|90)|5(?:0[0-4]|10|20|30|50|60|70|90)|6(?:00|20|"    .
                  "30|40|5[0156]|80)|7(?:0[0-29]|20|30|40|50|60|"            .
                  "7[01457])|8(?:0[0-9]|11)|9(?:00|1[0-35-9]|2[0-79]|"       .
                  "3[0-8]|4[3-9]|5[0-7]|6[0-9]|7[0-9]|80))|"                 .
              "6(?:0(?:0[0-589]|1[04-9]|2[0-3]|9[01])|1(?:0[0-36]|10|20|"    .
                  "30|40|50|60|70|90)|2(?:00|10|20|30|40|50|60|70)|"         .
                  "3(?:00|10|20|30|40|50|60)|400|500|600|700|800)|"          .
              "7(?:0(?:0[02-8]|13|2[03]|3[0-29])|1(?:10|20|30|40|50|60|"     .
                  "70|80|90)|2(?:0[0-57-9]|1[013]|20|30|40|50|60|70|90)|"    .
                  "3(?:0[0-57-9]|10|20|30|40|50|60|70|80|90)|4(?:00|10|"     .
                  "20|30|40|50|70|80)|5(?:0[0-2]|10|20|5[0-359])|600|700|"   .
                  "800|9(?:0[19]|1[0-8]|2[0-59]|31))|"                       .
              "8(?:00[0-35-9]|1(?:00|10|20|30|40|50|60|70|90)|2(?:00|10|"    .
                  "20|30|40|50|60|70)|3(?:00|10|20|30|40|70)|400|500|600|"   .
                  "700|800)|"                                                .
              "9(?:0(?:0[0-46-9]|1[014-8]|2[0-468]|3[2356]|4[13-79]|"        .
                  "5[1-359]|66|7[0-29]|80)|1(?:0[0-79]|1[0-25-79]|"          .
                  "2[02-5]|3[05-79]|40|50|60|70|8[0-4]|90)|2(?:20|30|"       .
                  "4[0-59]|50|60|70|80|90)|3(?:0[0-9]|1[0-249]|2[0157-9]|"   .
                  "30|40|50|60|70|80|90)|4(?:0[0-689]|1[0-578]|2[015-8]|"    .
                  "30|40|5[068]|60|8[0146]|90)|5(?:0[0-59]|10|20|30|40|"     .
                  "60|70|90)|6(?:0[0-29]|10|20|30|40|50|60|70|80|90)|"       .
                  "7(?:00|10|30|40|50|70)|80[0-59]|9(?:0[56]|1[1-39]|"       .
                  "2[57]|3[347-9]|4[19])))"                                  .

         "|5(?:0(?:0(?:0[0-9]|1[0-6]|50)|1(?:0[0-9]|1[03-5]|20|3[01]|40|"    .
                  "50|60|70|80|90)|2(?:0[0-9]|1[01]|20|30|40|50|60|70|"      .
                  "90)|3(?:0[0-9]|10|20|30|40|50|60|70|80|90)|"              .
                  "4(?:0[0-79]|10|20|30|4[0-24-9]|50|60|70|80|90)|5(?:00|"   .
                  "10|20|30|40|50|60|70|80|90)|6(?:00|10|20|30|40|5[12]|"    .
                  "60|70|80|90)|7(?:00|10|20|30|40|50|60|70)|8(?:00|10|"     .
                  "40|50|60|70|80|90)|95[018])|"                             .
              "1(?:0(?:0[015-9]|1[0-367]|2[12]|3[5-9]|4[1-3]|5[1-9]|"        .
                  "6[0-9]|7[0-79]|8[0-46-9]|9[025-79])|1(?:00|10|2[0-2]|"    .
                  "30|40|50|60|70|90)|2(?:0[0-9]|10|20|30|40|50|60|70|"      .
                  "90)|3(?:0[0-689]|1[089]|20|3[0-4]|4[09]|50|6[01]|70|"     .
                  "80|90)|4(?:0[01]|20|3[0-5]|50|60|70|80|90)|5(?:00|10|"    .
                  "20|30|7[1-3])|6(?:0[01]|6[2-4]|7[357-9]|8[2-9])|"         .
                  "7(?:00|1[35]|2[1-36])|8(?:0[01]|7[13]|8[46])|90[024])|"   .
              "2(?:0(?:0[0-8]|1[1246])|1(?:0[0-9]|1[0-35]|20|30|40|50|60|"   .
                  "70|90)|2(?:0[0-9]|10|20|30|40|50|60|70|90)|3(?:0[01]|"    .
                  "10|20|30|40|60|70)|4(?:00|10)|50[0-2]|600|700|800|"       .
                  "9(?:0[1-6]|1[579]|20))|"                                  .
              "3(?:0(?:0[0-9]|1[02-69]|2[0-2]|3[0-2]|6[0-39]|73|8[0-9]|"     .
                  "9[013-589])|1(?:0[0-59]|10|20|40|50|60|70|90)|"           .
                  "2(?:0[0-5]|10|20|30|40|50|60|70|90)|3(?:00|20|40|50|"     .
                  "60|70|80|90)|4(?:00|10|20|40|70|80)|5(?:00|40)|"          .
                  "6(?:0[0-2]|40)|700|8(?:00|1[01])|9(?:4[0-2]|5[01]|60|"    .
                  "7[01]))|"                                                 .
              "4(?:0(?:0[0-9]|1[0-7]|2[0-39]|3[125-9]|4[1-35-9]|5[2-46]|"    .
                  "6[2-4]|7[1-46]|82|9[6-8])|1(?:00|1[0-689]|2[0-39]|"       .
                  "3[0-689]|40|5[0-49]|60|70|8[0-69]|90)|2(?:0[0-69]|"       .
                  "1[01]|20|30|40|50|6[0-3]|7[0-39]|80|90)|3(?:0[0-59]|"     .
                  "1[0-29]|20|30|40|50|60|70|8[05]|90)|4(?:0[0-69]|"         .
                  "1[0-4]|2[05]|30|40|50|60|70|80|90)|5(?:0[0-79]|"          .
                  "1[0-79]|2[0-9]|30|40|50|60|70|80|90)|6(?:0[0-389]|10|"    .
                  "20|30|40|50|60|70|80|90)|7(?:0[0-689]|1[0-69]|20|30|"     .
                  "40|50|60|70|80|90)|8(?:0[0-3]|10|20|30|40|50|60|70|80|"   .
                  "90)|9(?:00|10|20|3[01349]|4[01]|50|60|70|80|90))|"        .
              "5(?:0(?:0[0-35-8]|1[2-5]|2[01])|1(?:0[0-9]|1[0239]|20|30|"    .
                  "40|50|60|70|90)|2(?:0[0-35]|10|2[01]|30|40|50|60|"        .
                  "7[01]|90)|3(?:00|10|20)|4(?:00|30)|500|600|700|8(?:00|"   .
                  "40))|"                                                    .
              "6(?:0(?:0[0-9]|1[06-9]|2[09]|3[46-9])|1(?:0[0-9]|10|20|30|"   .
                  "40|50|60|7[0-589]|90)|2(?:0[01479]|19|20|3[019]|40|50|"   .
                  "60|7[0-79]|90)|3(?:0[0-9]|1[0-279]|2[0-9]|30|4[0-39]|"    .
                  "50|60|70|8[0-3]|90)|4(?:0[0-79]|1[0139]|20|30|40|50|"     .
                  "60|70|80|90)|5(?:0[0-469]|10|20|3[0-2569]|40|50|60|70|"   .
                  "80|90)|6(?:0[0-379]|10|20|30|40|50|60|70|80|90)|"         .
                  "7(?:0[0-59]|30|40|50|60|70|80)|8(?:0[0-59]|40|5[045]|"    .
                  "60|70|80|9[0-369])|9(?:0[0-39]|10|20|30|5[014679]|"       .
                  "7[0379]|98))|"                                            .
              "7(?:0(?:0[03-9]|1[0-9]|2[0138]|3[0-8]|4[014-69]|5[0-4679]|"   .
                  "6[1-3]|7[0-9]|8[1-4])|1(?:0[0-689]|19|2[05-9]|3[0-3]|"    .
                  "4[0146-9]|5[0-579]|6[0-49]|7[05]|8[015]|9[0-39])|"        .
                  "2(?:0[0-9]|1[124-9]|20|3[0-4]|4[05]|5[05]|60|70|"         .
                  "8[0-39]|9[0-29])|3(?:0[0-49]|10|20|30|40|5[0-249]|"       .
                  "6[0-359]|7[0-39]|8[0-35]|90)|4(?:0[0-59]|1[025]|20|30|"   .
                  "4[05]|5[05]|60|70|80|90)|5(?:0[0-9]|1[05]|2[05]|3[05]|"   .
                  "40|50|6[05]|70|80|90)|6(?:0[0-9]|1[09]|20|3[05]|4[05]|"   .
                  "5[05]|60|70|8[05]|90)|7(?:0[0-59]|10|20|3[019]|4[078]|"   .
                  "5[167]|7[02478]|80|90)|8(?:0[013-59]|1[05]|20|30|40|"     .
                  "5[05]|6[05]|70|80|90)|9(?:05|1[0-3569]|2[05]|3[05]|40|"   .
                  "5[0-57-9]|60|7[0-489]|80|90))|"                           .
              "8(?:0(?:0[0-9]|1[3-9]|2[02-9]|3[3469])|1(?:10|20|30|40|50|"   .
                  "60|70|80|90)|2(?:0[0-9]|10|20|30|40|50|60|70|90)|"        .
                  "3(?:0[0-2]|10|20|30|40|50|60|70|80|90)|4(?:0[0-5]|10|"    .
                  "20|30|40|50|60|70|90)|5(?:0[0-38]|30)|6(?:00|4[0-39]|"    .
                  "60)|700|800)|"                                            .
              "9(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[1-9]|"           .
                  "6[0-3569]|7[0-489]|8[0-46-8]|9[127])|1(?:0[08]|1[0-9]|"   .
                  "2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|7[0-9]|8[0-24-9]|"     .
                  "9[0-589])|2(?:0[0237-9]|1[0-9]|2[0-79]|3[0-9]|4[0-79]|"   .
                  "5[0-589]|6[0-9]|7[0-57-9]|8[0-8]|9[02-79])|3(?:0[0-9]|"   .
                  "1[0-9]|2[0-2689]|3[0-9]|4[1569]|5[0189]|6[0-5]|"          .
                  "7[03-9]|8[0-3569]|9[0-39])|4(?:0[0-9]|1[0-369]|"          .
                  "2[0-24-9]|3[0-35-7]|4[0-9]|5[0-35-8]|6[0-8]|7[0-689]|"    .
                  "8[0-35-7]|9[0-7])|5(?:0[0-9]|10|2[02-49]|3[0-25-7]|"      .
                  "4[0-4]|5[0-57-9]|6[0-47-9]|7[0-3]|8[0-46-9]|90)|"         .
                  "6(?:0[0-8]|1[0-38]|20|3[0579]|40|5[0-9]|6[0-79]|"         .
                  "7[0-37]|8[0-35-79]|90)|7(?:0[0-9]|1[0-9]|2[0-367]|"       .
                  "3[0-5]|40|50|60|7[079]|8[0-57-9]|9[0-2])|8(?:00|"         .
                  "1[0-8]|20|3[0-289]|4[057-9]|5[0-39]|6[0-9]|7[0-8]|"       .
                  "8[0-579]|9[0-9])|9(?:00|10|20|3[0-3]|4[0-48]|5[0-2]|"     .
                  "6[0-57]|7[0-2569]|8[014]|9[08])))"                        .

         "|6(?:0(?:0(?:0[024-9]|1[0-9]|2[1-9]|3[15])|1(?:0[0-9]|1[0-9]|"     .
                  "2[0-36-9]|3[0-248]|4[019]|5[0357]|6[02]|7[035]|"          .
                  "8[013-5]|90)|2(?:0[0-9]|10|20|30|40|5[0-2]|60|7[01]|"     .
                  "8[01]|9[0-39])|3(?:0[0-9]|1[0-9]|2[0139]|3[02]|40|50|"    .
                  "60|70|80|90)|4(?:0[0-369]|10|20|3[0-7]|40|51|60|7[67]|"   .
                  "80|90)|5(?:0[0134689]|10|20|30|4[0-4]|5[0-2]|60|70|80|"   .
                  "90)|6(?:0[0237-9]|1[0-27-9]|20|3[1-57-9]|4[036-9]|50|"    .
                  "60|7[1247]|80|90)|7(?:00|1[01]|2[1-39]|3[0-24]|"          .
                  "4[0-59]|5[01]|6[125]|7[12679]|90)|8(?:0[0-5]|10|20|"      .
                  "3[1-3]|40|50|60|7[0-2]|80|90)|9(?:0[2389]|1[12]|"         .
                  "2[1-4]|3[0-2]|40|50|60|81))|"                             .
              "1(?:0(?:0[0-9]|1[1-9]|2[1-4]|4[12]|51)|1(?:0[0-79]|10|20|"    .
                  "30|40|50|60|70|90)|2(?:0[0-689]|10|20|30|40|50|60|70|"    .
                  "90)|3(?:0[0-6]|10|20|30|40|50|60|70|80|90)|4(?:00|10|"    .
                  "20|3[08]|4[08]|5[08]|70|90)|5(?:00|50|60|70)|600|"        .
                  "7(?:00|90)|8(?:00|91)|9(?:61|84|9[24]))|"                 .
              "2(?:0(?:0[0-57-9]|1[0-9]|2[0-9]|3[0-46]|5[1-59]|6[0-79]|"     .
                  "7[0-29]|8[019]|9[0-2])|1(?:0[0-9]|1[0-46-9]|2[0-46-9]|"   .
                  "3[0-246-9]|4[0-579]|5[0-35-9]|6[0-24-69]|7[023569]|"      .
                  "8[02579]|9[0-469])|2(?:0[0-79]|1[0-357-9]|2[0-68]|"       .
                  "3[0-2]|4[025-9]|5[0-9]|60|70|80|90)|3(?:0[0-9]|1[01]|"    .
                  "2[0-257]|3[0-6]|40|50|6[01]|70|80|90)|4(?:0[0-8]|"        .
                  "1[0-289]|20|30|40|5[0-39]|60|70|80|90)|5(?:0[0-9]|10|"    .
                  "20|30|40|50|60|7[05]|80|90)|6(?:0[0-4689]|10|20|30|40|"   .
                  "50|60|70|80|90)|7(?:0[0-29]|10|20|30|40|50|60|70|80|"     .
                  "90)|8(?:0[0-69]|10|20|30|40|50|60|70|8[0-29]|90)|"        .
                  "9(?:0[1-9]|10|2[0-39]|30|40|5[0-5]|6[05-9]|7[0-8]|80|"    .
                  "90))|"                                                    .
              "3(?:0(?:0[0-9]|1[0-9]|2[0-489]|3[0-9]|4[02-69]|5[014-8]|"     .
                  "6[34])|1(?:00|1[0-9]|2[02]|30|40|50|60|7[0-578]|90)|"     .
                  "2(?:0[0-6]|10|20|30|40|50|60|70|90)|3(?:0[0-8]|10|20|"    .
                  "30|40|50|60|70|80|90)|4(?:0[0-9]|10|20|30|40|50|60|70|"   .
                  "80|90)|5(?:0[0-6]|10|20|30|40|50|60|70|80|90)|6(?:00|"    .
                  "10|20|30|40|50|60|70|80|90)|7(?:00|10|20|30|40|50|60|"    .
                  "70|80|90)|8(?:0[0-48]|10|20|30|40|50|70|80|90)|9(?:00|"   .
                  "10|20|30|40|5[09]|6[0-9]|7[02]|80|90))|"                  .
              "4(?:0(?:0[0-46-8]|1[0-35-9]|2[0-47-9]|3[0-9]|4[0-46]|"        .
                  "5[0-9]|6[0-24]|7[1589]|8[0-2679]|90)|1(?:0[0-9]|"         .
                  "1[0-356]|2[0-2]|30|4[0-35-8]|50|60|70|8[1-57]|90)|"       .
                  "2(?:0[0-68]|10|20|3[0-8]|40|50|60|70|90)|3(?:0[0-47]|"    .
                  "10|20|30|40|50|60|70|90)|4(?:0[0-4]|10|20|30|40|50|60|"   .
                  "70|80|90)|5(?:0[0-6]|1[01]|20|30|60|70)|6(?:0[0-5]|40|"   .
                  "60|80)|7(?:0[0-5]|80)|8(?:00|11|70)|990)|"                .
              "5(?:0(?:0[0-9]|1[03-8]|2[0-5])|1(?:0[0-8]|1[0-267]|20|30|"    .
                  "40|50|7[01]|90)|2(?:0[0-4]|20|30|40|50|60|70|90)|"        .
                  "3(?:0[0-37-9]|1[0-2]|2[01]|30|50|60|70|80|90)|"           .
                  "4(?:0[0-2]|10|2[019]|30|40|60|90)|5(?:0[0-3]|10|60|"      .
                  "90)|6(?:0[01]|60|70|90)|7(?:0[016]|10)|80[01]|"           .
                  "9(?:0[1-57]|1[124578]|2[01]|30|5[01]))|"                  .
              "6(?:0(?:0[0-6]|1[1-57-9]|2[015-9]|3[0-49]|46|50|70)|"         .
                  "1(?:0[0-3]|1[0-3]|2[0-3]|30|4[0145]|50|6[0-25]|70|80|"    .
                  "90)|2(?:0[0-2]|10|20|30|40|50|60|70|80|90)|3(?:0[0-2]|"   .
                  "10|20|3[01459]|40|50|60|70|80|90)|4(?:0[0-3]|10|"         .
                  "2[0-3]|30|40|50|60|70|80|90)|5(?:0[0-29]|10|30|40|50|"    .
                  "60|70)|6(?:0[0-24-69]|10|20|5[0-29]|6[04]|70|80|90)|"     .
                  "7(?:0[0-4]|20|30|40|5[015]|60)|8(?:00|20|3[0-26]|45|"     .
                  "5[0346]|6[036]|70)|9(?:06|21|3[13]|4[015]|5[019]|"        .
                  "6[0-8]|87))|"                                             .
              "7(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[3-6]|59|6[01457-9]|"    .
                  "7[013-7]|8[0-9]|9[0-3689])|1(?:0[029]|1[02-8]|2[0-69]|"   .
                  "3[0-378]|4[0-6]|5[0-258]|6[0-35-79]|7[0-3]|90)|"          .
                  "2(?:0[0-79]|1[0-9]|20|3[0-2459]|4[0-39]|50|6[0-29]|70|"   .
                  "80|90)|3(?:0[0-8]|1[0-289]|20|30|40|50|60|70|8[0-2]|"     .
                  "90)|4(?:0[0-57-9]|1[0-2]|20|30|4[01]|5[0-9]|60|70|80|"    .
                  "90)|5(?:0[0-79]|10|2[01]|30|4[0-2]|50|60|70|80|90)|"      .
                  "6(?:0[0-9]|10|20|30|40|50|60|70|80|90)|7(?:0[0-9]|10|"    .
                  "2[0-8]|30|50|60|70|90)|8(?:0[0-479]|10|20|3[1-4689]|"     .
                  "4[0-3]|50|60|70|80|9[12])|9(?:0[05-79]|1[13]|"            .
                  "2[0-35-8]|3[0-3]|4[2-489]|5[2357-9]|6[0-79]|7[024-9]|"    .
                  "8[0-6]|9[09]))|"                                          .
              "8(?:0(?:0[0-9]|1[02-9]|2[0-79]|40|5[0-9]|6[0-9]|7[0-467]|"    .
                  "8[2-6]|9[0-379])|1(?:00|1[068]|2[014-8]|3[0-49]|40|"      .
                  "5[0-39]|60|7[0-3]|80|90)|2(?:00|10|2[01]|30|40|50|"       .
                  "6[0-5]|7[0-4]|80|90)|3(?:0[0-9]|1[0-7]|20|3[0-3]|40|"     .
                  "50|60|70|80|9[0-39])|4(?:00|10|20|40|60|70|80|90)|"       .
                  "5(?:0[0-49]|10|20|30|40|50|60|70|80|90)|6(?:00|10|20|"    .
                  "30|40|50|60|80|90)|7(?:0[0-5]|20|30|40|50|60|70|80|"      .
                  "90)|8(?:0[0-2469]|20|30|40|50|70|90)|9(?:10|2[013579]|"   .
                  "4[13-57-9]|50|6[08]|70|80|90))|"                          .
              "9(?:0(?:0[1-9]|6[124])|1(?:00|1[05]|2[04-69]|3[0-2469]|"      .
                  "4[0-578]|5[0-49]|6[01]|7[0-3]|9[0-259])|2(?:0[0-9]|"      .
                  "1[0-9]|2[05-9]|3[0279]|4[0-7]|5[0-35-9]|6[0-9]|"          .
                  "7[0-2569]|8[0-9]|9[0-4])|3(?:0[0-379]|1[0679]|"           .
                  "2[0-26-9]|3[06-9]|4[0-9]|5[0-9]|6[0-79]|7[0-3679]|"       .
                  "8[0459]|9[0-9])|4(?:0[0-689]|1[0-9]|2[0-9]|3[0-9]|"       .
                  "4[0-9]|5[0-9]|6[013-9]|7[024-9]|8[0-9]|9[0-35])|"         .
                  "5(?:0[0-359]|1[0-9]|20|3[01]|4[0-4]|5[0-3]|6[013-59]|"    .
                  "7[0-4689]|8[0-3]|9[0-5])|6(?:0[0-9]|1[0-3569]|2[0-9]|"    .
                  "3[0-9]|4[0-9]|5[0-9]|6[014579]|7[0-79]|8[024-79]|"        .
                  "9[0-46])|7(?:0[0-29]|14|2[0679]|3[0-25]|4[0-79]|5[15]|"   .
                  "60|7[01]|80|9[0-5])|8(?:0[0-689]|1[1-4679]|2[02-49]|"     .
                  "3[0-36]|4[01]|50|60|70|8[1-39]|9[01])|9(?:0[0-8]|10|"     .
                  "2[1-6]|3[0-9]|4[25-9]|5[1-57]|6[0-489]|70|98)))"          .

         "|7(?:0(?:0(?:0[0-7]|1[349]|2[0-2]|30)|1(?:0[0-4]|10|20|30|40|"     .
                  "50|60|70|80|90)|2(?:0[0-4]|10|20|30|40|50|70|80|90)|"     .
                  "3(?:0[0-46]|10|20|60)|4(?:00|40)|500|600|700|80[07])|"    .
              "1(?:0(?:0[0-9]|1[0-27-9]|2[0-9]|31|4[09])|1(?:0[0-9]|"        .
                  "1[08]|20|30|40|50|60|70|90)|2(?:0[0-69]|10|20|30|40|"     .
                  "50|60|70|90)|3(?:0[0-9]|10|2[0-8]|3[0-79]|40|50|60|70|"   .
                  "80|90)|4(?:0[0-57-9]|10|20|30|40|50|60|70|80|90)|"        .
                  "5(?:0[0-39]|10|20|30|40|50|70|80|90)|6(?:0[0-49]|20|"     .
                  "40|70|80)|7(?:00|10|40|60)|8(?:00|50|70|80)|9(?:60|"      .
                  "90))|"                                                    .
              "2(?:0(?:0[0-9]|1[013-689]|2[14-9]|3[02-9]|4[0-57-9]|"         .
                  "5[12589]|7[0-359]|8[0-9]|9[1-356])|1(?:0[09]|10|20|30|"   .
                  "40|50|60|70|90)|2(?:0[0-3568]|10|20|3[0-4679]|40|50|"     .
                  "60|70|90)|3(?:0[0-589]|10|20|30|40|50|60|70|80|90)|"      .
                  "4(?:0[0-79]|30|40|50|60|70)|5(?:00|10|30|40|50|60)|"      .
                  "6(?:00|10|50)|70[02-5]|800|90[28])|"                      .
              "3(?:0(?:0[0-9]|1[014-9]|2[0-24-6]|31|65|7[39]|89|9[14])|"     .
                  "1(?:0[0-9]|10|2[0-479]|30|40|5[0-579]|60|70|9[0-39])|"    .
                  "2(?:0[0-9]|1[0-2469]|2[013-59]|3[0-259]|40|50|"           .
                  "6[01459]|7[067]|9[0-2459])|3(?:0[0-39]|10|2[0-29]|30|"    .
                  "40|50|60|7[0-9]|8[12]|90)|4(?:0[0-39]|10|20|40|50|60|"    .
                  "70|80|9[0-49])|5(?:00|20|30|40|50|7[0-39]|90)|"           .
                  "6(?:0[0-49]|10|20|30|40|60|70)|7(?:0[0-589]|10|20|30|"    .
                  "90)|8(?:0[0-24-9]|70))|"                                  .
              "4(?:0(?:0[0-9]|1[0-6]|2[0145]|3[46]|4[01]|5[459])|"           .
                  "1(?:0[0-9]|1[0-3]|20|3[0-79]|40|5[0-9]|6[0-69]|70|90)|"   .
                  "2(?:0[0-9]|10|20|30|40|50|60|70|90)|3(?:0[0-57-9]|"       .
                  "1[014]|20|3[0-47-9]|40|50|60|7[0-4]|80|90)|"              .
                  "4(?:0[0-59]|10|20|30|40|50|60|70|80|90)|5(?:0[0-39]|"     .
                  "20|40|50|60|70|80)|6(?:0[0-79]|50|60)|7(?:0[0-469]|"      .
                  "40)|8(?:0[0-9]|90)|9(?:10|20|30|4[0-69]|5[0-4]|"          .
                  "6[0-469]|70|8[124-9]|9[013-9]))|"                         .
              "5(?:0(?:0[0-9]|1[0-9]|2[0134689]|3[0-689]|4[0-9]|5[1-689]|"   .
                  "6[0-9]|7[2-9]|8[0-9]|9[02-7])|1(?:0[0-9]|1[0-689]|"       .
                  "2[02-9]|3[01489]|4[0-24]|5[0-9]|6[1-9]|7[0-24679]|"       .
                  "8[0-24689]|9[124-8])|2(?:0[1-9]|1[0-9]|2[0-46-9]|"        .
                  "3[0-9]|4[01348]|5[0236]|6[0-69]|7[02589]|8[0-48]|"        .
                  "9[0-24589])|3(?:0[2-9]|1[0-57-9]|2[0-9]|3[0357-9]|"       .
                  "4[0-589]|5[0235-9]|6[0-7]|7[0-689]|8[0-9]|9[0-24-9])|"    .
                  "4(?:0[0-35-8]|1[0-35-9]|2[0-9]|3[0-24-9]|4[0-79]|"        .
                  "5[0-9]|6[02-8]|7[0-9]|8[0-8]|9[0-35-9])|5(?:0[2-9]|"      .
                  "1[0-2]|2[1-8]|3[3-8]|4[0-8]|5[0-8]|6[0-9]|7[0-9]|"        .
                  "8[0-9]|9[1-9])|6(?:0[0-9]|1[0-46-9]|2[0-9]|3[0-9]|"       .
                  "4[0-46-9]|5[0-9]|6[0-57-9]|7[013-57-9]|8[0-357-9]|"       .
                  "9[0-46-9])|7(?:0[0-35-9]|1[02-9]|2[0-79]|3[0-247-9]|"     .
                  "4[0-357-9]|5[03-79]|6[1-9]|7[0-358]|8[0-69]|9[04689])|"   .
                  "8(?:0[02-9]|1[2357-9]|2[0-9]|3[0-9]|4[013-9]|5[0-578]|"   .
                  "6[0-9]|7[0-24-9]|8[0-79]|9[0-57-9])|9(?:0[0-57-9]|"       .
                  "1[13-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-25-79]|7[0-9]|"   .
                  "8[013-9]|9[013-8]))|"                                     .
              "6(?:0(?:0[0-8]|1[127-9]|2[0-24-689]|3[0-35-9]|4[0-79]|"       .
                  "5[0-9]|6[0-9]|7[0-249]|8[03-7]|9[2-9])|1(?:0[017-9]|"     .
                  "1[013679]|2[0-579]|3[0-9]|4[0-4]|5[0-359]|6[014]|"        .
                  "7[0-9]|8[135-9]|9[0-7])|2(?:0[0-9]|10|20|3[0-35-9]|40|"   .
                  "50|60|70|80|90)|3(?:0[0-68]|10|20|30|40|50|60|7[09]|"     .
                  "80|90)|4(?:0[0-59]|1[089]|20|30|40|50|60|70|80|90)|"      .
                  "5(?:0[0-49]|10|20|30|40|50|60|70|80|90)|6(?:00|10|20|"    .
                  "30|40|50|60|80|90)|7(?:00|10|2[013]|30|40|50|60|70|80|"   .
                  "90)|8(?:0[0-9]|10|2[13-5]|40|50|60|70|8[0-57]|90)|"       .
                  "9(?:0[02]|1[0457]|2[05-79]|3[045]|4[05]|50|60|70|80))|"   .
              "7(?:0(?:0[0-9]|1[0-9]|2[013-8]|39|4[0-3]|5[0-2]|90)|"         .
                  "1(?:0[0-9]|1[0-58]|2[02-467]|3[013-589]|4[01458]|"        .
                  "5[0147]|6[03-79]|7[01346-8]|8[13-7]|9[0-9])|"             .
                  "2(?:0[0-9]|1[0-9]|2[0-59]|3[0-249]|4[0-36-9]|5[0-579]|"   .
                  "6[0-46]|7[0-2]|8[0-2]|9[0-9])|3(?:0[0-57-9]|1[0-689]|"    .
                  "2[07]|3[0-57]|4[0-46-9]|5[0-3]|60|70|8[0-24589]|90)|"     .
                  "4(?:0[0-579]|1[0-79]|2[0-79]|3[0-79]|4[0-8]|5[0-58]|"     .
                  "6[02-57-9]|70|8[0-46-9])|5(?:0[0-9]|1[05]|2[0-9]|"        .
                  "4[0-9]|5[0-9]|6[0-9]|70|8[0-35]|90)|6(?:0[0-9]|"          .
                  "1[0-24-7]|20|30|4[01569]|50|60|70|80|90)|7(?:0[0-689]|"   .
                  "1[014-7]|2[0-3]|30|50|60|7[126-9]|80|9[1-689])|"          .
                  "8(?:1[013-9]|20|3[0-57-9]|40|50|60|7[0-3569]|80|90)|"     .
                  "9(?:10|2[03]|3[0-39]|40|5[01]|61|70|8[1-369]|90))|"       .
              "8(?:0(?:0[0-9]|1[0-57-9]|2[0-246-9]|3[02-5]|4[1-9]|"          .
                  "5[1-79]|6[0-24-7]|7[189]|8[1-9]|9[1-69])|"                .
                  "1(?:0[0-24-689]|1[0-9]|2[0-24-69]|3[0-59]|4[0-35-9]|"     .
                  "5[0-57-9]|6[02-79]|7[0-479]|8[0-689]|9[0-9])|"            .
                  "2(?:0[0-9]|1[0249]|2[0-39]|3[0-49]|4[0-49]|50|60|70|"     .
                  "8[0-9]|9[0-49])|3(?:0[0-46-9]|1[0-79]|2[0-2]|3[019]|"     .
                  "4[0-24-69]|5[0-69]|6[0-48]|7[0-9]|80|9[015])|"            .
                  "4(?:0[0-49]|1[0-35-79]|2[0-49]|3[019]|40|5[0178]|60|"     .
                  "7[0-29]|80|90)|5(?:0[0-79]|1[0-59]|20|3[0-69]|40|50|"     .
                  "60|70|80|9[0-29])|6(?:0[0-69]|1[029]|20|30|40|50|60|"     .
                  "70|80|90)|7(?:0[0-359]|1[0-58]|20|30|40|50|60|7[0189]|"   .
                  "80|90)|8(?:0[0-69]|10|20|30|40|5[0-249]|60|70|8[1-6]|"    .
                  "9[07])|9(?:1[05]|2[0-9]|3[01]|4[0-689]|5[05]|60|70|80|"   .
                  "9[05-7]))|"                                               .
              "9(?:0(?:0[0-9]|1[0-8]|2[1-9]|3[0-9]|4[1-69]|5[0-69]|"         .
                  "6[0189]|7[0-79]|8[0-8]|9[1-389])|1(?:0[0-69]|10|20|30|"   .
                  "4[0-5]|50|60|70|8[0-2459]|90)|2(?:0[0-689]|10|20|"        .
                  "3[0-4]|40|50|60|70|90)|3(?:0[0-9]|10|20|30|40|50|60|"     .
                  "70|80|90)|4(?:0[0-49]|10|20|30|40|50|60)|5(?:00|10)|"     .
                  "600|700|800|9(?:39|5[03])))"                              .

         "|8(?:0(?:0(?:0[0-9]|1[0-79]|2[0-35-9]|3[0-46-9]|4[0-46-9]|"        .
                  "5[014-7]|6[0-489]|75|8[0-9]|9[0-79])|1(?:0[0-9]|"         .
                  "1[058]|2[02]|3[0-24-6]|4[02-6]|50|60|70|90)|"             .
                  "2(?:0[0-689]|10|20|30|40|50|60|70|90)|3(?:0[0-39]|10|"    .
                  "20|3[0-69]|40|50|60|70|90)|4(?:00|10|20|30|40|50|60|"     .
                  "70|80|90)|5(?:00|10|20|3[1-59]|40|50|60|70|80)|6(?:00|"   .
                  "10|20|30|40|50|70|80|90)|7(?:00|10|40|50|70|80)|"         .
                  "8(?:00|20|30|50|60|70|80|9[01])|9(?:1[09]|60|70|80))|"    .
              "1(?:0(?:0[0-7]|1[1-9]|2[0-35-8]|3[014-9]|90)|1(?:0[0-9]|"     .
                  "1[056]|20|30|40|50|60|70|90)|2(?:0[0-9]|1[05]|20|30|"     .
                  "40|50|60|70|90)|3(?:0[0-5]|10|20|30|40|50|60|70|80|"      .
                  "90)|4(?:00|30|40|50|70|90)|5(?:0[0-369]|30|40|70|80)|"    .
                  "6(?:0[0-59]|30|40|60)|7(?:00|10)|800|990)|"               .
              "2(?:0(?:0[0-689]|1[3-579]|2[47]|3[0237]|4[078]|5[347]|"       .
                  "6[057]|7[07]|8[07])|1(?:0[0-49]|10|20|30|40|50|60|70|"    .
                  "90)|2(?:0[0-28]|10|20|30|40|50|70|90)|3(?:0[0-3]|30|"     .
                  "40|50|60|70|90)|4(?:0[0-3]|10|40)|500|600|7(?:00|10)|"    .
                  "800)|"                                                    .
              "3(?:0(?:0[0-8]|1[2-69]|3[068]|4[0-289]|5[0-9]|6[0-59]|"       .
                  "7[016-9]|8[0-9]|9[0-57-9])|1(?:0[078]|1[019]|20|"         .
                  "3[016]|4[039]|50|6[0-467]|7[0-79]|8[013-57-9]|"           .
                  "9[0-269])|2(?:00|10|20|3[016-8]|40|50|60|70)|3(?:00|"     .
                  "1[0-7]|20|30|40|50|7[0-29]|80|90)|4(?:0[0-9]|1[128]|"     .
                  "20|30|40|60|70|8[0-9]|90)|5(?:0[0-479]|1[0-248]|20|30|"   .
                  "50|60|70|80|90)|6(?:0[0-4689]|1[03-68]|30|40|60|70|80|"   .
                  "90)|7(?:0[0-57]|20|40|80|90)|8(?:00|20|30|40|60|70|"      .
                  "90)|9(?:10|20|5[1-47]|80|9[0-8]))|"                       .
              "4(?:0(?:0[04-9]|1[013-689]|2[1-579]|3[1-6]|4[1-9]|"           .
                  "5[014689]|7[1278]|8[1-5]|9[0-9])|1(?:0[0-9]|10|2[0-4]|"   .
                  "3[0-59]|4[0-479]|50|60|70|90)|2(?:0[0-9]|10|20|3[0-2]|"   .
                  "40|50|60|7[0-9]|90)|3(?:0[0-9]|10|20|30|40|50|60|70|"     .
                  "80|90)|4(?:0[0-5]|10|20|30|40|50|60|70|80|90)|"           .
                  "5(?:0[0-57]|10|30|50|60|70|80)|6(?:0[0-39]|60)|"          .
                  "7(?:0[0-6]|40|50|60)|8(?:0[0-58]|10|20|30|40|50|60|"      .
                  "7[0-48])|9(?:0[1259]|1[1-9]|5[1-46]|6[13-57-9]|"          .
                  "7[1-367]))|"                                              .
              "5(?:0(?:0[0-9]|1[0235-9]|2[01346]|3[356])|1(?:0[0-9]|"        .
                  "1[013489]|20|3[0-39]|40|50|6[0-579]|70|80|90)|"           .
                  "2(?:0[0-689]|10|20|30|40|50|60|70|80|9[0-29])|"           .
                  "3(?:0[0-79]|10|20|30|4[0-2]|50|60|70|90)|"                .
                  "4(?:0[0-4679]|10|20|30|40|50|60|70|80|90)|5(?:0[0-69]|"   .
                  "10|20|30|40|50|60|70|80|90)|6(?:0[0-479]|1[02-6]|20|"     .
                  "30|40|60|70|80|90)|7(?:0[0237-9]|10|40|50|70)|"           .
                  "80[03-69]|9(?:19|2[1-8]|3[0-4]))|"                        .
              "6(?:0(?:0[0-9]|1[0-3]|2[0-39]|3[04-68]|4[2-7]|5[0-6]|"        .
                  "6[0-35-9]|7[1-46]|8[0-9]|9[0-8])|1(?:0[0-9]|1[07]|20|"    .
                  "3[0-57-9]|40|50|60|70|80|90)|2(?:0[0-2469]|10|20|30|"     .
                  "40|50|60|70|8[0-29]|90)|3(?:00|10|20|30|40|50|6[0-29]|"   .
                  "70|80|90)|4(?:00|10|20|30|40|50|60|70|80|90)|"            .
                  "5(?:0[0-289]|10|30|40|50|80)|6(?:00|10)|700|800|"         .
                  "9(?:05|10|6[0-49]|8[0-4]))|"                              .
              "7(?:0(?:0[0-9]|1[0-35-8]|2[0-356]|3[0-46-9]|4[0-9]|"          .
                  "5[0-589]|6[015-9]|7[013-9]|8[0-59]|90)|1(?:00|10|20|"     .
                  "30|40|50|60|70|90)|2(?:0[0-356]|10|2[0-3]|30|40|50|60|"   .
                  "70|80|90)|3(?:00|10|20|30|40|50|60|70|80)|4(?:00|10|"     .
                  "20|30|40|60|70|80)|5(?:00|10|20|70|90)|6(?:00|20|40)|"    .
                  "7(?:00|20)|8(?:00|90)|9(?:00|19|20|30|50))|"              .
              "8(?:0(?:0[0-9]|1[0-8]|2[015-79]|33|5[01]|60|8[4-689]|99)|"    .
                  "1(?:0[0-9]|10|2[0-3579]|3[0-39]|4[0-39]|5[0-6]|"          .
                  "6[0-259]|70|87|9[0-589])|2(?:0[0-79]|1[0-4]|20|30|40|"    .
                  "50|60|70|90)|3(?:0[0-7]|10|20|30|40|50|60|70|80|90)|"     .
                  "4(?:0[0-35-9]|10|20|30|40|50|60|70|80|90)|"               .
                  "5(?:0[0-479]|10|20|30|40|50|60|80)|6(?:00|30|40|50)|"     .
                  "700|80[0-59])|"                                           .
              "9(?:0(?:0[0-79]|1[0-2569]|2[0-9]|30|89|9[0-59])|"             .
                  "1(?:0[0-9]|1[036]|20|30|4[04]|50|60|70|90)|"              .
                  "2(?:0[0-79]|10|20|30|40|50|60|70|90)|3(?:0[0-79]|10|"     .
                  "20|3[01]|40|50|60|80|90)|4(?:00|10|20|30|40|50|60|70|"    .
                  "80)|5(?:00|1[05]|2[05]|30|50|60|70|80)|6(?:00|30|60|"     .
                  "90)|7(?:00|10|40|7[0-2])|800))"                           .

         "|9(?:0(?:0(?:0[0-9]|1[0-689]|2[0-35]|40)|1(?:0[01]|10|20|30|40|"   .
                  "50|60|70)|200|3(?:00|30|40|50|60|70|80)|400|500|600|"     .
                  "700|8(?:00|50))|"                                         .
              "1(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|"           .
                  "7[0-247-9]|80|9[07])|1(?:0[0-9]|2[0-9]|3[0-35-7]|"        .
                  "4[0-269]|5[0-46-9]|6[0-9]|7[0-47-9]|80|9[0-9])|"          .
                  "2(?:0[014579]|1[0134]|2[0-9]|30|4[01]|50|6[0-35-9]|70|"   .
                  "80|9[0-24-9])|3(?:0[0-9]|1[0-245]|2[0-589]|3[0-59]|"      .
                  "4[0-9]|5[013-59]|60|7[014]|8[0-9]|90)|4(?:0[0-79]|"       .
                  "1[024-8]|2[0-9]|30|40|50|6[0-3]|70|80|90)|5(?:10|20|"     .
                  "30|4[0-29]|5[01]|60|7[0-489]|80|90)|6(?:0[0-359]|"        .
                  "1[0239]|2[05]|30|40|50|60|70|80|9[0-9])|7(?:0[0-9]|"      .
                  "1[0-3]|20|30|4[0-9]|50|6[0-47]|70|8[013]|9[0-2])|"        .
                  "8(?:0[0-79]|1[0-5]|2[01]|30|4[019]|5[0-25]|6[01]|"        .
                  "7[0-9]|8[0-79]|9[0-689])|9(?:1[0-79]|2[1-4]|30|4[0-9]|"   .
                  "5[1-9]|6[1-9]|7[1-9]|8[1-68]))|"                          .
              "2(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[3-9]|4[0-9]|5[0-9]|"           .
                  "6[013-9]|7[0-9]|8[1-9]|9[235-9])|1(?:0[0-689]|1[0-9]|"    .
                  "2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|7[0-9]|8[1-689]|"      .
                  "9[013-69])|2(?:0[0-689]|1[0-69]|2[0-9]|3[0-57-9]|"        .
                  "4[0-79]|5[0-46-9]|6[0-369]|7[0-59]|8[0-246-8]|"           .
                  "9[0-35-8])|3(?:0[0-9]|1[0-46-9]|2[0-9]|3[0-379]|"         .
                  "4[0-2]|5[0-79]|6[0-69]|7[029]|8[08]|9[0-9])|"             .
                  "4(?:0[0-9]|1[0-9]|2[03]|3[08]|4[1-578]|54|99)|"           .
                  "5(?:0[0-9]|1[1346-9]|2[1-57-9]|3[1-8]|4[1-6]|5[235]|"     .
                  "6[1-79]|7[13-8]|8[1-8]|9[1-9])|6(?:0[0-79]|1[13-6]|"      .
                  "2[24]|3[1-8]|4[1-359]|5[013-689]|6[05]|7[12579]|"         .
                  "8[2-689]|9[1358])|7(?:0[0-9]|1[1-5]|2[125-9]|3[05-9]|"    .
                  "4[1-578]|5[12568]|6[1-46-8]|7[2-469]|8[1-9]|9[1-9])|"     .
                  "8(?:0[0-9]|1[135-9]|2[0-4]|4[2-8]|5[1-69]|6[1-7]|"        .
                  "8[1-79]|9[1-8])|9(?:0[1-46-9]|1[1-9]|2[0-9]|3[0-9]|59|"   .
                  "7[0-57-9]|8[013-9]|99))|"                                 .
              "3(?:0(?:0[0-35-9]|1[1-46-8]|2[124])|1(?:0[0-8]|1[0-24-8]|"    .
                  "2[0-367]|3[013-6]|4[0-9]|5[0-8]|6[0-9]|7[0-24-7]|"        .
                  "8[1-46-9]|9[0-79])|2(?:0[0-9]|1[0-46-9]|2[0-2]|"          .
                  "3[0135]|4[015]|50|6[01]|7[0145]|8[2-6]|9[047])|"          .
                  "3(?:0[0-8]|1[015]|2[0-5]|3[0-24-7]|4[0-689]|5[0-2]|"      .
                  "6[0-6]|70|8[0-2]|90)|4(?:0[0-8]|1[0-2]|2[0-4]|3[0167]|"   .
                  "4[01]|5[013-8]|6[0-5]|7[0239]|8[1-79]|93)|5(?:0[0-8]|"    .
                  "1[1-7]|2[1-46-8]|3[1-9]|4[1457]|5[4-68]|6[1-4]|"          .
                  "7[14-8]|8[1-9]|9[13])|6(?:0[0-689]|1[1-9]|2[1-37]|31|"    .
                  "9[1-9])|7(?:0[015]|1[129]|3[1346-8]|61)|8(?:0[0-267]|"    .
                  "1[2-6]|8[13-57]|9[19])|90[17-9])|"                        .
              "4(?:0(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[1-5]|94)|"       .
                  "1(?:0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-69]|"   .
                  "7[0-5]|9[0-79])|2(?:0[0-9]|1[0149]|2[0-9]|3[0-9]|"        .
                  "4[0-4679]|5[0-9]|6[0-9]|7[0-9]|81|90)|3(?:0[0-9]|"        .
                  "1[0-29]|2[0-247]|4[0-59]|5[013-79]|6[0-689]|7[0-9]|"      .
                  "8[0-9]|9[0-689])|4(?:0[0-9]|1[0-579]|20|3[0-9]|40|"       .
                  "5[0-79]|60|7[0-9]|80|90)|5(?:0[0-9]|1[0-9]|2[0-8]|"       .
                  "3[1-9]|4[1-46-9]|5[0-3]|6[1-9]|7[1-9]|8[1-9]|9[1-9])|"    .
                  "6(?:0[0-4679]|1[1-46-9]|2[1-36-8]|3[1-9]|4[1-578]|"       .
                  "5[1346-9]|6[13479]|7[1-9]|8[1-8])|7(?:0[0-4679]|"         .
                  "1[0-59]|2[1-9]|3[1-9]|4[1-589]|6[1-8]|8[1-59])|"          .
                  "8(?:0[0-9]|1[1-9]|3[1-39]|5[1-9]|6[1-79]|80|9[1-4])|"     .
                  "9(?:4[125-9]|5[1-689]|6[1-68]|7[1-8]))|"                  .
              "5(?:0(?:0[0-8]|1[0-9]|2[0-57-9]|3[0-8]|4[0-26]|5[0-69]|"      .
                  "6[0-79]|7[0-9]|9[0-8])|1(?:0[0-9]|1[0-27-9]|2[0-79]|"     .
                  "3[0-579]|4[0-6]|5[0-57-9]|6[0-259]|70|80|9[0-9])|"        .
                  "2(?:0[0-79]|1[0-9]|2[0-4689]|3[0-49]|40|5[0-29]|60|70|"   .
                  "80|90)|3(?:0[0-469]|1[0-689]|2[0-489]|3[0-5]|40|"         .
                  "5[015]|60|7[09]|80|90)|4(?:00|1[0-29]|20|30|4[015]|50|"   .
                  "6[01]|7[0-9]|80|90)|5(?:0[0-9]|10|2[0-9]|30|40|5[019]|"   .
                  "60|7[013]|80|90)|6(?:0[0-59]|1[0-9]|20|30|40|5[0-29]|"    .
                  "60|70|80|9[0-367])|7(?:0[0-9]|1[0-3569]|2[0-7]|3[1-4]|"   .
                  "4[0-27]|50|6[019]|70|80)|8(?:0[0-9]|1[0-6]|20|30|"        .
                  "4[02-4]|50|6[1-68]|7[0-79]|80|9[127])|9(?:0[12578]|"      .
                  "1[1-3569]|2[0-8]|3[0-4]|4[0-9]|5[0-46-9]|7[0-8]))|"       .
              "8799)"                                                        .
    ")",


    #
    # 5 Digit postal code, with leading 0s.
    #
    # Data from: http://download.geonames.org/export/zip/DE.zip
    # 
    Germany     =>
      "(?k:0(?:1(?:0(?:6[79]|9[79])|1(?:0[89]|2[79]|39|5[679]|69|8[79])|"    .
                  "2(?:1[79]|3[79]|5[79]|7[79])|3(?:0[79]|14|2[468])|"       .
                  "4(?:45|5[48]|6[258]|7[178])|5(?:58|61|8[79]|9[14])|"      .
                  "6(?:09|1[269]|23|40|6[25]|8[39])|7(?:05|2[38]|3[1478]|"   .
                  "44|6[28]|7[3468]|96)|8(?:09|1[469]|2[459]|33|4[478]|"     .
                  "55|77|96)|9(?:0[0469]|17|20|36|45|68|79|8[37]|"           .
                  "9[03468]))|"                                              .
              "2(?:6(?:2[57]|33|4[03]|8[19]|9[249])|7(?:08|27|3[0369]|"      .
                  "4[278]|63|79|8[258]|9[14679])|8(?:2[6-9]|9[49])|"         .
                  "9(?:06|2[39]|43|5[3679]|7[79]|9[1479]))|"                 .
              "3(?:0(?:4[2468]|5[0-58]|9[69])|1(?:03|1[69]|3[09]|49|59|"     .
                  "72|85|97)|2(?:0[25]|2[269]|38|4[69]|53))|"                .
              "4(?:1(?:0[3579]|29|5[57-9]|7[7-9])|2(?:0[579]|29|49|"         .
                  "7[579]|8[89]|99)|3(?:1[5-9]|2[89]|4[79]|5[67])|4(?:16|"   .
                  "2[05]|35|42|51|6[03])|5(?:09|19|23|39|52|6[457]|"         .
                  "7[1459])|6(?:0[03]|1[0378]|26|39|43|5[1457]|68|"          .
                  "8[03578])|7(?:03|20|36|4[169]|58|69|7[49])|8(?:08|"       .
                  "2[1478]|38|49|6[02]|74|8[069]|95)|9(?:1[06]|2[48]|"       .
                  "3[12468]))|"                                              .
              "6(?:1(?:08|1[02468]|2[02468]|3[02]|79|8[48]|9[38])|2(?:17|"   .
                  "3[17]|4[269]|5[4589]|68|79|95)|3(?:08|1[1378]|33|"        .
                  "4[378]|6[69]|8[568])|4(?:0[68]|2[059]|49|5[68]|"          .
                  "6[34679]|84|93)|5(?:0[27]|2[68]|3[67]|4[2378]|56|67|"     .
                  "7[178])|6(?:18|28|3[268]|4[278]|67|79|8[268])|7(?:12|"    .
                  "2[124579]|31|49|66|7[349]|8[056]|9[1246])|"               .
                  "8(?:0[03489]|4[24679]|6[129]|8[689]|9[56])|9(?:0[159]|"   .
                  "1[78]|2[2568]))|"                                         .
              "7(?:3(?:18|3[03468]|4[39]|56|6[68]|8[179])|4(?:07|"           .
                  "2[2679])|5(?:4[5689]|5[1247]|70|8[069])|6(?:07|1[369]|"   .
                  "29|39|46)|7(?:4[3579]|51|68|7[48])|8(?:06|19)|9(?:07|"    .
                  "19|2[24679]|37|5[02578]|73|8[0579]))|"                    .
              "8(?:0(?:5[68]|6[0246])|1(?:07|1[258]|3[24]|4[147])|2(?:09|"   .
                  "2[38]|3[3679]|48|58|6[12579]|8[09]|9[47])|3(?:0[149]|"    .
                  "1[258]|2[14-68]|4[049]|5[259]|7[13]|9[36])|4(?:12|"       .
                  "2[78]|3[29]|5[19]|68|85|9[169])|5(?:2[3579]|3[89]|"       .
                  "4[1378])|6(?:06|26|4[58]))|"                              .
              "9(?:0(?:28|30)|1(?:1[1-4679]|2[0235-8]|3[01])|2(?:1[27]|"     .
                  "2[1478]|3[256]|4[1349])|3(?:06|2[268]|37|5[0356]|66|"     .
                  "76|8[057]|9[02459])|4(?:05|19|2[379]|3[02579]|56|"        .
                  "6[58]|7[147]|8[1478]|96)|5(?:09|1[478]|26|4[48]|57|69|"   .
                  "7[3579]|99)|6(?:0[03]|1[89]|2[379]|3[348]|48|6[19])))"    .

         "|1(?:0(?:1(?:1[579]|7[89])|24[3579]|3(?:1[57-9]|6[579])|"          .
                  "4(?:0[579]|3[579])|5(?:5[13579]|8[579])|62[3579]|"        .
                  "7(?:0[79]|1[13579]|7[79]|8[13579])|82[3579]|"             .
                  "9(?:6[13579]|9[79]))|"                                    .
              "1011|2(?:0(?:4[3579]|5[13579]|99)|1(?:0[13579]|5[79]|"        .
                  "6[13579])|2(?:0[3579]|4[79]|7[79])|3(?:0[579]|4[79]|"     .
                  "5[13579])|4(?:3[579]|59|8[79])|5(?:2[4679]|5[579]|"       .
                  "8[79])|6(?:19|2[1379]|79|8[13579]))|"                     .
              "3(?:0(?:47|5[13579]|8[689])|1(?:2[579]|5[689]|8[79])|"        .
                  "3(?:4[79]|5[13579])|4(?:0[3579]|3[579]|6[579])|"          .
                  "5(?:0[3579]|8[13579]|9[13579])|62[79])|"                  .
              "4(?:0(?:5[023579]|89)|1(?:09|29|31|6[3579]|9[3579])|"         .
                  "4(?:6[179]|7[1368]|8[02])|5(?:13|32|4[278]|5[02478])|"    .
                  "6(?:12|2[147]|41|56|6[29])|7(?:1[25]|2[78]|7[02468]|"     .
                  "89|9[378])|8(?:06|2[2378])|9(?:13|29|4[37]|59|7[49]))|"   .
              "5(?:2(?:3[0246]|9[59])|3(?:06|2[0468]|4[45]|66|7[02478])|"    .
                  "5(?:1[78]|2[68]|37|6[269])|7(?:11|3[28]|4[15689]|"        .
                  "5[124578])|8(?:06|27|3[1478]|48|59|6[48]|9[08])|"         .
                  "9(?:07|1[03]|26|3[68]))|"                                 .
              "6(?:2(?:2[57]|30|4[478]|59|69|78)|3(?:0[367]|21|4[18]|"       .
                  "5[269])|5(?:15|4[078]|5[269]|6[257])|7(?:27|6[167]|75|"   .
                  "9[28])|8(?:1[68]|27|3[1357]|45|6[68])|9(?:09|18|2[18]|"   .
                  "4[59]))|"                                                 .
              "7(?:0(?:3[3469]|8[79]|9[1489])|1(?:09|11|2[169]|39|5[349]|"   .
                  "6[68]|79|9[24])|2(?:0[79]|1[3479]|3[57]|48|5[2589]|68|"   .
                  "79|91)|3(?:09|2[12689]|3[57]|4[89]|58|67|7[359]|89|"      .
                  "9[0-28])|4(?:06|19|2[49]|38|4[09]|5[49]|89|9[1358])|"     .
                  "50[69])|"                                                 .
              "8(?:0(?:5[579]|69)|1(?:0[679]|19|4[67]|8[124]|9[0568])|"      .
                  "2(?:09|11|25|3[01369]|4[69]|58|7[369]|9[29])|"            .
                  "3(?:1[147]|20|3[47]|47|56|7[45])|4(?:3[579]|4[25]|"       .
                  "6[159])|5(?:07|1[0369]|28|46|5[16]|6[59]|7[34]|8[16])|"   .
                  "609)|"                                                    .
              "9(?:0(?:5[3579]|6[13579]|7[13579]|8[69])|2(?:0[59]|17|30|"    .
                  "4[369]|58|60|73|88|94)|3(?:0[0369]|22|3[69]|48|57|"       .
                  "7[0246]|86|9[59])|4(?:06|1[27])))"                        .

         "|2(?:0(?:0(?:38|88|9[579])|14[4689]|2(?:49|5[13579])|35[04579]|"   .
                  "45[79]|53[579])|"                                         .
              "1(?:0(?:29|3[13579]|7[3579])|1(?:0[79]|29|4[79])|"            .
                  "2(?:1[78]|2[0478]|44|5[5689]|6[16]|7[1249])|"             .
                  "3(?:3[579]|5[478]|6[0589]|7[169]|8[02568]|9[14578])|"     .
                  "4(?:0[013679]|23|3[5689]|4[124579]|65|8[13]|93)|"         .
                  "5(?:0[29]|1[46]|2[124679])|6(?:14|29|35|4[0134679]|"      .
                  "8[02-4]|98)|7(?:0[269]|1[0247]|2[03679]|3[02479]|45|"     .
                  "5[56]|6[2359]|7[0256]|8[124579]))|"                       .
              "2(?:0(?:4[13579]|8[13579])|1(?:1[13579]|4[3579]|59|"          .
                  "7[579])|29[79]|3(?:0[13579]|3[579]|59|9[13579])|"         .
                  "4(?:1[579]|5[3579])|5(?:2[3579]|4[79]|59|8[79])|"         .
                  "60[579]|76[13579]|8(?:4[468]|5[01]|69|8[059])|"           .
                  "9(?:2[679]|4[169]|5[25689]|6[24579]))|"                   .
              "3(?:5(?:39|5[2468]|6[024689]|70)|6(?:1[179]|2[36-9]|69|"      .
                  "8[349])|7(?:01|1[4579]|3[08]|4[34679]|58|69|7[4579]|"     .
                  "95)|8(?:1[23568]|2[0134679]|4[357]|58|6[03679]|79|"       .
                  "8[13]|9[689])|9(?:09|1[19]|23|36|4[2468]|52|6[68]|"       .
                  "7[024]|9[269]))|"                                         .
              "4(?:1(?:0[35-79]|1[134689]|4[35-9]|59|61)|2(?:1[147]|"        .
                  "2[02369]|3[2589]|4[124578]|5[0134679])|3(?:06|2[1679]|"   .
                  "40|5[1478]|6[0134679]|76|9[2589])|40[124579]|"            .
                  "5(?:3[4679]|58|68|76|8[29]|9[48])|6(?:01|1[0369]|"        .
                  "2[0235689]|3[124578]|4[0134679])|7(?:68|8[2-47]|"         .
                  "9[0134679])|8(?:0[0235689]|1[134679]|37|48|5[0257]|"      .
                  "6[0134679]|7[0235689]|8[124578]|9[0134679])|9(?:3[79]|"   .
                  "4[134]|55|6[0369]|7[25-7]|8[03689]|9[124679]))|"          .
              "5(?:3(?:3[5-7]|48|5[58]|6[1458]|7[013679])|4(?:04|2[19]|"     .
                  "3[67]|51|6[29]|7[49]|8[25689]|9[124579])|5(?:24|4[18]|"   .
                  "5[147]|6[0369]|7[235689]|8[124578]|9[0134679])|693|"      .
                  "7(?:0[49]|1[2589]|2[14579]|46|6[147]|7[0469]|8[2568]|"    .
                  "9[124579])|8(?:13|2[16]|3[26]|4[0259]|5[0235689]|"        .
                  "6[02-4689]|7[0235689]|8[124579]|99)|9(?:17|2[03467]|"     .
                  "38|46|80|9[2679]))|"                                      .
              "6(?:1(?:2[1-3579]|3[135]|6[09]|8[08]|97)|2(?:0[39]|1[59])|"   .
                  "3(?:16|4[059]|8[24689])|4(?:09|19|27|34|4[16]|52|65|"     .
                  "74|8[679])|5(?:06|2[49]|32|48|5[36]|7[19])|6(?:0[357]|"   .
                  "2[49]|3[29]|55|7[06]|8[39])|7(?:2[135]|36|5[79]|89)|"     .
                  "8(?:02|1[07]|26|3[15]|4[24579]|71|9[279])|"               .
                  "9(?:0[134679]|19|3[15-79]|54|69))|"                       .
              "7(?:2(?:11|3[29]|4[35689]|5[12479]|83|99)|3(?:0[58]|1[38]|"   .
                  "2[147]|3[03679]|56|67|74|8[369])|4(?:04|1[29]|32|"        .
                  "4[269]|7[2468]|9[89])|5(?:68|7[02468]|80)|6(?:07|"        .
                  "1[269]|2[48]|3[278])|7(?:11|2[169]|49|5[135]|77|"         .
                  "9[38])|80[149])|"                                         .
              "8(?:19[579]|2(?:0[13579]|1[13579]|3[79]|59|7[79])|"           .
                  "3(?:0[79]|2[579]|35|5[579])|7(?:1[79]|5[579]|7[79]|"      .
                  "90)|8(?:16|32|44|57|65|7[069]))|"                         .
              "9(?:22[13579]|3(?:0[38]|13|2[038]|3[169]|4[258]|"             .
                  "5[1-35689]|6[124579]|7[89]|8[69]|9[2-469])|4(?:1[036]|"   .
                  "39|5[169]|6[258]|7[1-35689]|8[124578]|9[0134679])|"       .
                  "5(?:25|49|5[369]|6[258]|7[14-689]|8[124578]|"             .
                  "9[0134679])|6(?:14|33|4[0369]|64|83|9[039])))"            .

         "|3(?:0(?:1(?:59|6[13579]|7[13579])|4(?:19|49|5[13579])|5(?:19|"    .
                  "21|39|59)|6(?:2[579]|5[579]|69)|8(?:2[367]|5[135]|80|"    .
                  "90)|9(?:00|16|26|38|52|66|74|8[29]))|"                    .
              "1(?:0(?:08|2[089]|3[23569]|61|7[39]|8[457-9]|9[134679])|"     .
                  "1(?:3[4579]|41|57|6[27]|7[147]|8[058]|9[1569])|"          .
                  "2(?:2[468]|34|4[169]|75)|3(?:03|1[19])|5(?:15|35|"        .
                  "4[27]|5[235689]|82|9[25])|6(?:0[034689]|1[389]|"          .
                  "2[1-36-9]|3[2-46-8]|55|75|8[38]|9[1389])|7(?:0[0278]|"    .
                  "1[0-2457-9]|37|49|8[579])|8(?:12|32|4[08]|55|6[0378]))|"  .
              "2(?:0(?:49|5[12])|1(?:0[578]|20|3[09])|2(?:57|78|89)|"        .
                  "3(?:12|39|51|6[19])|4(?:2[3579]|57|69|79)|5(?:4[579]|"    .
                  "84)|6(?:0[29]|57|76|8[39]|9[49])|7(?:5[68]|60|91)|"       .
                  "8(?:05|16|25|3[29]))|"                                    .
              "3(?:0(?:14|3[49]|98)|1(?:0[0246]|29|42|54|6[15]|7[58]|"       .
                  "8[149])|3(?:11|3[0245]|78|97)|4(?:15|28|4[29])|519|"      .
                  "6(?:0[24579]|1[13579]|4[79]|59|89|99)|7(?:19|29|39|58|"   .
                  "75|90)|8(?:03|1[38]|2[49]))|"                             .
              "4(?:1(?:1[79]|2[13578]|3[0-24])|2(?:12|25|33|46|53|6[06]|"    .
                  "7[07]|8[169]|9[258])|3(?:0[258]|1[147]|2[03679]|46|"      .
                  "5[59]|69|7[69]|8[58]|9[369])|4(?:14|3[149]|54|66|"        .
                  "7[1479]|97)|5(?:08|1[369]|37|49|60|76|8[27]|9[0369])|"    .
                  "6(?:13|2[168]|3[02379]))|"                                .
              "5(?:0(?:3[79]|4[13]|66|75|8[0358]|9[1469])|1(?:0[248]|"       .
                  "1[024679])|2(?:16|3[269]|60|7[49]|8[2578])|3(?:05|15|"    .
                  "2[1579]|9[02468])|4(?:1[058]|2[38]|35|4[047]|5[27]|"      .
                  "6[0369])|5(?:1[069]|7[689]|8[0-6])|6(?:06|1[49]|25|"      .
                  "3[038]|4[1479]|8[3-9]|90)|7(?:08|1[369]|45|5[369]|"       .
                  "6[478]|8[19]|9[2469]))|"                                  .
              "6(?:0(?:3[79]|4[13]|88|93)|1(?:0[03]|1[059]|2[49]|3[27]|"     .
                  "4[258]|5[147]|6[03679]|79|99)|2(?:0[58]|1[1479]|51|"      .
                  "6[69]|7[257]|8[024679])|3(?:04|18|2[03569]|41|5[58]|"     .
                  "6[479]|81|9[169])|4(?:04|1[49]|33|48|5[267]|6[069]))|"    .
              "7(?:0(?:7[3579]|8[135])|1(?:15|2[047]|3[0369]|54|7[06]|"      .
                  "8[16]|9[1479])|2(?:1[3-8]|35|4[279]|69|76|8[147]|"        .
                  "9[03679])|3(?:08|18|27|39|45|5[159])|4(?:12|3[14]|"       .
                  "4[14579])|5(?:20|3[49]|47|74|8[169])|6(?:03|19|2[07]|"    .
                  "3[2359]|4[02379]|71|88|9[1679]))|"                        .
              "8(?:023|1(?:0[02468]|1[02468]|2[0246]|5[49]|6[25]|"           .
                  "7[0369])|2(?:2[689]|39|59|68|7[124579]|81)|3(?:0[024]|"   .
                  "1[259]|2[124579]|50|6[48]|7[235689]|8[124578])|"          .
                  "4(?:4[02468]|5[89]|6[124578]|7[0134679]|8[69])|5(?:18|"   .
                  "2[478]|3[01369]|4[237]|5[0134679])|6(?:4[024]|67|78|"     .
                  "85|90)|7(?:0[0479]|2[39])|8(?:2[0289]|3[568]|55|"         .
                  "7[1579]|89|9[59]))|"                                      .
              "9(?:014|1(?:0[468]|1[02468]|2[02468]|30|6[47]|7[159])|"       .
                  "2(?:18|21|4[059]|6[14]|79|88|91)|3(?:07|1[79]|26|"        .
                  "4[035]|5[69]|65|87|9[378])|4(?:18|3[459]|4[3468])|"       .
                  "5(?:17|24|39|7[69]|9[069])|6(?:06|1[59]|2[49]|38|"        .
                  "4[69])))"                                                 .

         "|4(?:0(?:2(?:1[0-3579]|2[13579]|3[13579])|4(?:68|7[024679]|89)|"   .
                  "5(?:4[579]|89|9[13579])|6(?:2[579]|6[78]|70|99)|"         .
                  "7(?:2[134]|64|89)|8(?:22|32|78|8[0235]))|"                .
              "1(?:06[135689]|1(?:69|79|89|99)|23[689]|3(?:34|52|6[36]|"     .
                  "7[29])|4(?:6[024689]|7[02])|5(?:1[5-7]|39|4[0-2]|"        .
                  "6[49])|7(?:4[7-9]|51)|8(?:12|36|4[49]))|"                 .
              "2(?:1(?:0[3579]|1[13579])|2(?:7[579]|8[13579])|3(?:2[79]|"    .
                  "49|69|89|99)|4(?:77|89|99)|5(?:49|5[135]|79)|"            .
                  "6(?:5[13579]|9[79])|7(?:19|81|99)|8(?:5[3579]|9[79])|"    .
                  "929)|"                                                    .
              "4(?:1(?:3[579]|4[13579])|2(?:2[579]|6[3579]|8[79])|3(?:09|"   .
                  "19|2[89]|39|5[79]|69|79|88)|5(?:3[246]|7[579]|81)|"       .
                  "6(?:2[357-9]|49|5[1-3])|7(?:8[79]|9[13579])|"             .
                  "8(?:0[13579]|6[679]|79|9[24]))|"                          .
              "5(?:1(?:2[78]|3[0134689]|4[13-579])|2(?:19|39|5[79]|"         .
                  "7[679]|89)|3(?:0[79]|2[679]|5[5-79])|4(?:03|68|"          .
                  "7[0235689]|81)|5(?:2[579]|49)|6(?:5[79]|6[135]|99)|"      .
                  "7(?:01|11|21|3[19]|68|7[02])|8(?:79|8[134689]|"           .
                  "9[124679])|96[468])|"                                     .
              "6(?:04[579]|1(?:1[79]|4[579])|2(?:3[68]|4[024]|8[246])|"      .
                  "3(?:25|4[28]|5[49]|9[579])|4(?:1[49]|46|59|8[357]|99)|"   .
                  "5(?:09|1[49]|3[579]|6[29]))|"                             .
              "7(?:05[1357-9]|1(?:19|3[7-9]|6[679]|7[89]|9[89])|"            .
                  "2(?:2[689]|39|49|59|69|79)|4(?:4[1357]|75|95)|"           .
                  "5(?:0[69]|33|46|5[19]|74|89)|6(?:08|2[3-7]|38|47|52|"     .
                  "6[159])|79[89]|8(?:0[02-579]|29|39|77)|9(?:06|18|29))|"   .
              "8(?:079|1(?:4[3579]|5[13579]|6[1357])|2(?:31|49|68|82|91)|"   .
                  "3(?:0[18]|17|2[49]|36|4[16]|5[16]|6[169])|4(?:29|"        .
                  "3[12]|55|65|77|8[058]|9[369])|5(?:2[79]|31|65|99)|"       .
                  "6(?:07|1[29]|2[49]|53|83|91)|7(?:03|12|2[07]|3[49]))|"    .
              "9(?:0(?:7[468]|8[02468]|90)|1(?:24|34|43|52|63|7[069]|86|"    .
                  "9[16])|2(?:0[15]|1[49])|3(?:2[468]|56|77|93)|"            .
                  "4(?:0[16]|1[39]|2[49]|3[49]|48|5[13679]|7[79]|9[27])|"    .
                  "5(?:0[49]|25|36|4[59]|65|77|8[46]|9[34679])|6(?:10|"      .
                  "2[46]|3[2578]|61|8[158]|9[269])|7(?:16|33|4[04]|5[17]|"   .
                  "6[27]|7[0479])|8(?:0[89]|11|2[48]|3[258]|4[34679])))"     .

         "|5(?:0(?:1(?:2[679]|69|7[01]|8[19])|2(?:26|59)|3(?:21|54|74|"      .
                  "89)|6(?:6[78]|7[0246-9])|7(?:3[3579]|6[579])|"            .
                  "8(?:2[3579]|5[89])|9(?:3[13579]|6[89]|9[679]))|"          .
              "1(?:06[13579]|1(?:0[3579]|4[3579])|3(?:7[13579]|81|99)|"      .
                  "4(?:2[79]|6[579]|91)|5(?:03|1[59]|45|70|8[08]|9[78])|"    .
                  "6(?:4[357]|74|88)|7(?:0[29]|66|89))|"                     .
              "2(?:0(?:6[2468]|7[02468]|80)|1(?:34|46|5[269])|2(?:2[2-4]|"   .
                  "49)|3(?:49|5[135]|7[29]|8[258]|9[1369])|4(?:28|4[15]|"    .
                  "5[79]|77|99)|5(?:11|25|3[18]))|"                          .
              "3(?:1(?:1[13579]|2[13579]|7[3579])|22[579]|3(?:32|4[037]|"    .
                  "59)|4(?:2[46]|74|89|98)|5(?:0[15-8]|18|20|3[349]|"        .
                  "4[57]|57|6[027]|7[27-9])|6(?:04|19|39)|7(?:21|57|73|"     .
                  "83|97)|8(?:0[49]|19|4[024]|59|79|81|94)|9(?:0[29]|"       .
                  "1[39]|25|37|4[0579]))|"                                   .
              "4(?:29[02-68]|3(?:0[689]|1[01346-8]|2[09]|3[128]|"            .
                  "4[0134679])|4(?:1[13]|2[124679]|39|41|5[0135-79]|"        .
                  "7[02]|8[3467]|9[278])|5(?:1[68]|2[34689]|3[134689]|"      .
                  "5[028]|68|7[04689]|8[4-79]|9[57])|6(?:08|1[0-24679]|"     .
                  "3[46]|4[679]|5[57]|6[24689]|7[35]|8[79]))|"               .
              "5(?:1(?:1[68]|2[0246-9]|3[01])|2(?:18|3[2479]|46|5[27]|"      .
                  "6[238]|7[0168]|8[368]|9[1469])|4(?:1[13]|2[245]|"         .
                  "3[0257]|4[24]|5[0279]|69|71|8[137]|9[014679])|"           .
                  "5(?:4[356]|59|6[689]|7[168]|8[35]|9[0235-79])|"           .
                  "6(?:0[68]|1[89]|2[14679])|7(?:43|5[68]|6[578]|"           .
                  "7[4679]))|"                                               .
              "6(?:0(?:68|7[0235-7])|1(?:12|3[023]|54|7[09]|82|91)|"         .
                  "2(?:0[346]|18|20|3[57]|4[249]|5[34]|69|7[16]|8[138]|"     .
                  "9[01459])|3(?:0[57]|1[67]|2[1-39]|3[023578]|4[01689]|"    .
                  "5[57]|68|7[079])|4(?:1[024]|2[2478]|5[79]|62|7[0279])|"   .
                  "5(?:6[467]|7[59]|8[147-9]|9[3489])|6(?:26|3[07]|"         .
                  "4[258]|5[1369])|7(?:2[79]|36|4[356]|5[1349]|6[1679])|"    .
                  "8(?:1[248]|2[0135689]|4[13]|5[0689]|6[124579]))|"         .
              "7(?:0(?:7[2468]|80)|2(?:23|34|5[08]|71|9[09])|3(?:19|"        .
                  "3[49]|68|9[29])|4(?:13|39|62|8[29])|5(?:18|20|3[79]|"     .
                  "48|55|6[27]|7[278]|8[0134679])|6(?:1[024]|2[79]|"         .
                  "3[25689]|4[124578]))|"                                    .
              "8(?:0(?:89|9[13579])|1(?:19|35)|2(?:39|56|85)|3(?:00|13|"     .
                  "3[29])|45[2-6]|5(?:0[79]|1[135]|40|53|66|79)|"            .
                  "6(?:3[68]|4[024]|75)|7(?:0[68]|10|3[09]|6[29]|91)|"       .
                  "8(?:0[29]|4[09]))|"                                       .
              "9(?:0(?:6[3579]|7[1357])|1(?:74|9[29])|2(?:2[79]|69)|"        .
                  "3(?:02|2[09]|48|68|79|87|9[49])|4(?:2[357]|39|57|69|"     .
                  "94)|5(?:05|1[049]|5[5-8]|81|9[07])|60[29]|75[579]|"       .
                  "8(?:2[13]|46|72|89)|9(?:09|29|39|55|6[49])))"             .

         "|6(?:0(?:3(?:08|1[13468]|2[0235-79]|8[5689])|4(?:3[1357-9]|"       .
                  "8[6-9])|5(?:2[89]|49|9[4689]))|"                          .
              "1(?:1(?:18|3[078]|69|84|9[147])|2(?:0[0369]|3[19]|50|67|"     .
                  "7[369])|3(?:48|5[02]|8[19])|4(?:4[09]|62|7[69]))|"        .
              "3(?:0(?:6[579]|7[135])|1(?:10|28|50|65|79)|2(?:25|63)|"       .
                  "3(?:03|2[29])|4(?:05|5[02467]|77|86)|5(?:0[05]|1[27]|"    .
                  "26|3[38]|4[369]|7[19]|8[49]|9[49])|6(?:07|19|28|"         .
                  "3[3679]|54|67|7[49]|8[38]|9[14579])|7(?:39|4[13]|55|"     .
                  "6[28]|7[36]|85|9[16])|8(?:0[18]|1[14]|2[05689]|3[149]|"   .
                  "4[0369]|5[367]|6[0347-9]|7[124579]|97)|9(?:06|1[16]|"     .
                  "2[04578]|3[0134679]))|"                                   .
              "4(?:2(?:8[3579]|9[1357])|3(?:19|31|4[27]|54|67|72|8[05]|"     .
                  "9[057])|40[14579]|5(?:21|46|6[09]|7[29]|8[49])|6(?:25|"   .
                  "46|5[38]|6[58]|7[38]|8[369])|7(?:11|20|3[29]|4[37]|"      .
                  "5[034679])|8(?:07|23|3[29]|46|5[039]))|"                  .
              "5(?:1(?:8[3579]|9[13579])|2(?:0[1357]|19|2[0-4]|3[29])|"      .
                  "3(?:07|2[169]|4[3-7]|66|75|8[58]|9[169])|4(?:28|39|51|"   .
                  "6[28]|7[49])|5(?:10|2[079]|49|5[0-68]|8[29]|9[479])|"     .
                  "6(?:0[46]|1[148]|2[034679])|7(?:19|60|79|95)|"            .
                  "8(?:1[27]|24|3[05]|43)|9(?:29|3[1346]))|"                 .
              "6(?:1(?:1[13579]|2[135-9]|3[0-3])|2(?:65|71|8[07]|9[29])|"    .
                  "3(?:33|46|5[29]|86|99)|4(?:24|40|5[039]|8[24]|97)|"       .
                  "5(?:0[0134679]|3[89]|40|57|64|7[18]|8[39])|6(?:06|"       .
                  "2[059]|36|4[069]|63|79|87|93)|7(?:0[169]|40|63|73|"       .
                  "8[07]|9[38])|8(?:0[269]|22|39|49|51|6[29]|7[179]|"        .
                  "8[257]|9[24])|9(?:0[13479]|1[4679]|5[3-57]|69|7[68]|"     .
                  "8[179]|9[469]))|"                                         .
              "7(?:0(?:59|6[13579]|71|98)|1(?:05|1[27]|2[25-7]|3[346]|"      .
                  "4[1679]|5[027-9]|6[15-79])|2(?:2[79]|4[056]|5[1689]|"     .
                  "69|7[138]|8[013]|9[2457])|3(?:0[4578]|1[01679]|46|54|"    .
                  "6[013568]|7[346-8])|4(?:3[3-5]|5[49]|6[68]|7[1-35]|"      .
                  "8[02379])|5(?:4[79]|5[01]|7[4578]|8[0235-7]|"             .
                  "9[0-35689])|6(?:5[3579]|6[13]|7[78]|8[01568]|9[1379])|"   .
                  "7(?:0[015-7]|1[4-68]|2[2457-9]|3[12457]|4[24-689]|"       .
                  "5[2-4679])|8(?:0[68]|1[134679]|2[1-4679]))|"              .
              "8(?:1(?:59|6[13579]|99)|2(?:19|29|39|59)|30[579]|5(?:19|"     .
                  "26|35|4[29])|6(?:23|4[279])|7(?:23|53|66|75|8[29]|"       .
                  "9[49])|80[49])|"                                          .
              "9(?:1(?:1[578]|2[01346]|51|68|81|9[08])|2(?:07|14|2[16]|"     .
                  "3[149]|4[25]|5[0134679])|4(?:12|2[79]|3[4679]|69|"        .
                  "8[38]|93)|5(?:0[29]|1[478])))"                            .

         "|7(?:0(?:1(?:7[3468]|8[02468]|9[0-3579])|3(?:2[79]|7[2468])|"      .
                  "4(?:3[579]|69|99)|5(?:6[3579]|9[79])|6(?:19|29)|"         .
                  "7(?:3[46]|71|94)|8(?:06|25|39))|"                         .
              "1(?:0(?:3[24]|6[3579]|8[38]|93)|1(?:0[16]|1[16]|2[06]|"       .
                  "3[149]|4[49]|5[4579])|2(?:29|54|63|7[27]|8[27]|"          .
                  "9[2679])|3(?:3[246]|64|84|9[47])|40[49]|5(?:22|"          .
                  "4[0369]|54|6[036]|7[03679])|6(?:3[468]|4[02]|65|7[29]|"   .
                  "86|9[16])|7(?:0[16]|1[178]|2[0369]|3[2579]))|"            .
              "2(?:07[0246]|1(?:08|1[69]|2[47]|3[158]|4[14579]|60|7[258]|"   .
                  "8[1469])|2(?:02|1[38]|2[14679]|50|7[05]|8[05]|"           .
                  "9[0134679])|3(?:36|48|5[15689]|6[124579]|79|93)|"         .
                  "4(?:0[16]|1[14579]|5[89]|6[19]|7[4579]|88)|5(?:0[15]|"    .
                  "1[0134679]|25|3[124579]|55|74|8[124579])|6(?:22|"         .
                  "3[169]|4[49]|5[4578]|6[0134679])|7(?:6[02468]|70|93)|"    .
                  "8(?:0[05]|1[038]|2[079]))|"                               .
              "3(?:0(?:3[357]|54|6[16]|7[29]|8[47]|9[2589])|"                .
                  "1(?:0[124578]|1[0134679])|2(?:07|3[05]|4[09]|5[27]|"      .
                  "6[25689]|7[124578])|3(?:12|2[69]|3[37]|4[024579])|"       .
                  "4(?:3[0-4]|4[17]|5[037]|6[03679]|79|8[5689]|"             .
                  "9[124579])|5(?:2[579]|4[07]|5[037]|6[035689]|"            .
                  "7[124579])|6(?:14|3[05]|42|5[05]|6[03679])|7(?:28|"       .
                  "3[02-4]|6[05]|7[0369]))|"                                 .
              "4(?:0(?:7[2468]|8[01])|1(?:7[27]|8[29]|9[369])|2(?:06|"       .
                  "1[149]|2[369]|3[2589]|4[35689]|5[124579])|3(?:21|"        .
                  "3[67]|4[38]|5[47]|6[0369]|7[2469]|8[2589]|9[124579])|"    .
                  "4(?:05|17|2[034679])|5(?:23|3[258]|4[124579]|64|"         .
                  "7[259]|8[2569]|9[24579])|6(?:13|2[69]|3[2589]|53|"        .
                  "7[03679])|7(?:06|22|3[16]|4[034679])|8(?:21|3[1248]|"     .
                  "4[27]|5[058]|6[124579]|89)|9(?:0[69]|1[258]|2[14578]|"    .
                  "3[0134679]))|"                                            .
              "5(?:0(?:15|3[18]|45|5[03679])|1(?:7[23579]|8[01]|96)|"        .
                  "2(?:03|1[07]|2[38]|3[369]|4[2589])|3(?:05|2[38]|"         .
                  "3[14579]|65|78|8[2579]|9[124579])|4(?:17|28|3[38]|"       .
                  "4[3679]))|"                                               .
              "6(?:006|1(?:3[13579]|49|8[579]|99)|2(?:2[7-9]|75|87|97)|"     .
                  "3(?:07|16|27|3[27]|44|5[169])|4(?:37|48|56|6[17]|"        .
                  "7[034679])|5(?:3[024]|4[79]|71|9[3679])|6(?:46|6[19]|"    .
                  "76|8[49]|9[48])|7(?:0[3679]|26|44|5[16]|6[1478]|"         .
                  "7[0134679])|8(?:29|3[135]|4[68]|5[57]|6[35]|7[0279]|"     .
                  "8[79]|91))|"                                              .
              "7(?:6(?:5[246]|94)|7(?:0[49]|16|2[38]|3[16]|4[0369]|56|"      .
                  "6[17]|7[036]|8[147]|9[0134679])|8(?:15|3[0369]|55|66|"    .
                  "7[16]|8[03679])|9(?:33|4[48]|55|6[036]|7[124578]))|"      .
              "8(?:0(?:48|5[0246]|7[38]|8[3679]|98)|1(?:12|2[06]|3[26]|"     .
                  "4[1478]|66|76|8[37]|9[49])|2(?:24|3[49]|4[47]|5[0369]|"   .
                  "6[2679])|3(?:15|3[37]|4[35]|5[14579])|4(?:6[2457]|"       .
                  "7[69])|5(?:32|49|5[49]|6[47]|7[0369]|8[0235689]|"         .
                  "9[124578])|6(?:0[0134679]|28|47|5[258]|6[124579])|"       .
                  "7(?:13|27|3[03679]))|"                                    .
              "9(?:098|1(?:0[02468]|1[0-2457]|8[39]|9[49])|2(?:06|1[159]|"   .
                  "2[47]|3[258]|4[149]|5[2468]|6[138]|7[1469]|8[0235689]|"   .
                  "9[124579])|3(?:12|3[16]|4[168]|5[0369]|6[124579]|79|"     .
                  "95)|4(?:00|1[058]|2[34679])|5(?:13|39|4[01]|7[68]|"       .
                  "8[589]|9[124579])|6(?:18|39|50|6[49]|7[47]|8[235689]|"    .
                  "9[124579])|7(?:13|25|3[03679]|61|7[147]|8[07]|9[038])|"   .
                  "8(?:0[124579]|22|37|4[38]|5[369]|6[258]|7[124579])))"     .

         "|8(?:0(?:33[135-79]|469|53[89]|6(?:3[46-9]|8[679])|79[6-9]|"       .
                  "80[1-579]|9(?:3[3579]|9[23579]))|"                        .
              "1(?:24[13579]|3(?:69|7[13579])|47[5-79]|5(?:39|4[13579])|"    .
                  "6(?:6[79]|7[13579])|73[579]|82[579]|92[579])|"            .
              "2(?:0(?:08|24|3[12]|4[19]|5[47]|6[14579])|1(?:10|31|40|52|"   .
                  "66|78|94)|2(?:05|1[16]|2[39]|3[479]|56|6[69]|7[25689]|"   .
                  "8[124578]|9[0134679])|3(?:19|27|35|4[03679]|62|77|"       .
                  "8[03679]|9[0235689])|4(?:0[124579]|18|3[1-35689]|"        .
                  "4[124579]|67|8[178]|9[0134679])|5(?:15|38|4[1479]))|"     .
              "3(?:0(?:2[246]|43|5[29]|64|7[15]|8[038]|9[38])|1(?:0[149]|"   .
                  "1[259]|2[235689]|3[124579])|2(?:09|2[49]|3[36]|4[26]|"    .
                  "5[034679]|78)|3(?:0[18]|1[37]|2[49]|3[49]|4[2469]|"       .
                  "5[2589]|6[124578]|7[0134679]|95)|4(?:04|1[0367]|35|"      .
                  "5[1478]|71|8[3679])|5(?:12|27|3[0369]|4[34679]|"          .
                  "5[0235689]|6[124579])|6(?:07|2[034679]|46|6[16]|"         .
                  "7[0134679]|84)|7(?:0[0378]|14|27|3[04579]))|"             .
              "4(?:0(?:28|3[0246]|48|5[16]|6[169]|7[269]|8[2589]|"           .
                  "9[124578])|1(?:0[0134679]|3[07]|4[049]|5[25]|"            .
                  "6[034689]|7[124578]|8[0134679])|3(?:07|2[369]|"           .
                  "3[23579]|47|59|6[47]|7[158]|8[14579])|4(?:05|1[69]|"      .
                  "2[478]|3[124579]|53|78|89|94)|5(?:0[38]|1[38]|2[49]|"     .
                  "3[39]|4[34679]|5[0235689]|6[124578]|7[0134679]))|"        .
              "5(?:0(?:49|5[1357]|7[27]|8[048]|9[258])|1(?:0[147]|"          .
                  "1[0134679]|2[0235689]|3[124579])|2(?:2[19]|3[258]|"       .
                  "4[147]|5[034689]|76|83|9[0368])|3(?:0[124579]|5[46]|"     .
                  "68|7[56]|86|9[159])|4(?:0[2568]|1[0134679]|35|4[57]|"     .
                  "5[2679]|6[124579])|5(?:21|40|51|6[07]|7[09]|86|"          .
                  "9[189])|6(?:0[49]|1[47]|2[25]|3[05]|4[0369]|5[235689]|"   .
                  "6[124579])|7(?:16|37|48|57|64|7[478]))|"                  .
              "6(?:1(?:5[02-4679]|6[13579]|79|99)|3(?:16|43|56|68|81|"       .
                  "9[19])|4(?:05|15|2[04]|38|4[147]|5[0369]|6[25]|"          .
                  "7[03679]|8[0235689]|9[124578])|5(?:0[024578]|"            .
                  "1[013479]|29|5[14689]|6[124578]|7[0134679])|6(?:09|"      .
                  "3[37]|4[137]|5[0357]|6[03689]|7[2-689]|8[124578]|"        .
                  "9[024578])|7(?:0[0134679]|20|3[235689]|4[124578]|"        .
                  "5[0134679])|8(?:07|25|3[036]|4[25]|5[034679]|"            .
                  "6[0235689]|7[124579]|99)|9(?:1[169]|2[0235689]|"          .
                  "3[124578]|4[0134679]|56|7[124578]|8[0134679]))|"          .
              "7(?:4(?:3[579]|48|5[29]|6[36]|7[147]|8[0478]|9[034679])|"     .
                  "5(?:09|27|3[48]|4[14579]|61)|6(?:00|16|29|3[47]|"         .
                  "4[02578]|5[0134679]|6[0235689]|7[124579])|7(?:00|19|"     .
                  "2[47]|3[034679]|4[0235689]|5[124578]|6[0134679]|"         .
                  "7[0235689]|8[124579]))|"                                  .
              "8(?:0(?:4[568]|69|7[49]|8[56]|9[0479])|1(?:3[18]|4[2579]|"    .
                  "6[17]|7[1589])|2(?:1[2-4]|39|5[05]|6[037]|7[1369]|"       .
                  "8[14579]|99)|3(?:1[679]|26|39|48|5[36]|6[1478]|"          .
                  "7[0134679])|4(?:00|1[06]|2[27]|3[0367]|4[1478]|"          .
                  "5[0134679]|7[17]|8[0134679]|99)|5(?:1[258]|2[14579])|"    .
                  "6(?:05|3[0134679]|62|77|82|9[03679])|7(?:09|1[89]))|"     .
              "9(?:0(?:40|7[3579]|81)|1(?:29|34|43|5[05]|6[058]|"            .
                  "7[134679]|8[0235689]|9[124578])|2(?:3[13]|5[07]|6[49]|"   .
                  "7[58]|8[147]|9[0134679])|3(?:12|3[15]|4[034679]|"         .
                  "5[0235689]|6[124578])|4(?:07|15|2[03689]|3[124578]|"      .
                  "4[0134679])|5(?:18|2[02]|37|4[27]|5[158]|6[1478]|84|"     .
                  "97)|6(?:0[14578]|1[0134679])))"                           .

         "|9(?:0(?:4(?:0[2389]|1[19]|2[579]|3[19]|4[139]|5[13579]|6[19]|"    .
                  "7[1358]|8[029]|91)|5(?:1[38]|22|3[07]|4[27]|5[269]|62|"   .
                  "7[149]|8[47]|9[269])|6(?:0[27]|1[034679])|76[23568])|"    .
              "1(?:0(?:5[2468]|7[47]|8[03568]|9[0134679])|1(?:26|54|"        .
                  "6[16]|7[147]|8[03679])|2(?:07|17|2[047]|3[035689]|"       .
                  "4[124579]|57|7[58]|8[124679])|3(?:01|15|2[0257]|"         .
                  "3[02468]|4[14679]|5[0235689]|6[124579])|4(?:13|38|"       .
                  "4[38]|5[269]|6[0235689]|7[124578]|8[0134679])|5(?:22|"    .
                  "41|5[05]|6[047]|7[258]|8[03679]|9[0235689])|"             .
                  "6(?:0[124578]|1[0134679]|2[0235689]|3[124579])|"          .
                  "7(?:1[079]|2[0235689]|3[124578]|4[0134679]|57|8[158]|"    .
                  "9[0235689])|80[124579])|"                                 .
              "2(?:2(?:24|37|4[259]|5[369]|6[0235689]|7[124578]|"            .
                  "8[0134679])|3(?:18|3[149]|4[258]|5[358]|6[0134679])|"     .
                  "4(?:21|3[169]|4[24579])|5(?:07|2[16]|3[369]|"             .
                  "4[0235689]|5[124579])|6(?:37|48|55|6[05]|7[06]|8[15]|"    .
                  "9[034679])|7(?:0[0235689]|1[124578]|2[0134679]))|"        .
              "3(?:0(?:4[79]|5[13579]|7[37]|8[03679]|9[0235689])|"           .
                  "1(?:0[124579]|28|3[38]|4[29]|5[258]|6[147]|7[03679]|"     .
                  "8[0235689]|9[124579])|3(?:09|26|3[369]|4[235689]|"        .
                  "5[124689])|4(?:13|26|37|4[49]|5[358]|6[2468]|"            .
                  "7[0134679]|8[0235689]|9[124579]))|"                       .
              "4(?:0(?:3[246]|51|6[05]|7[28]|8[169]|9[49])|1(?:0[47]|"       .
                  "1[0368]|2[147]|3[03679]|4[0235689]|5[124578]|"            .
                  "6[0134679])|2(?:09|27|3[49]|4[49]|5[0235689]|"            .
                  "6[124579])|3(?:15|27|3[0369]|4[24578]|5[0134679]|"        .
                  "6[0235689]|7[124579])|4(?:05|19|2[48]|3[1679]|47|69|"     .
                  "74|8[16]|9[16])|5(?:0[158]|1[38]|2[2679]|3[0235689]|"     .
                  "4[124578]|5[0134679]|6[0235689]|7[124579]))|"             .
              "5(?:0(?:28|3[02])|1(?:00|1[19]|26|3[18]|45|5[28]|6[38]|"      .
                  "7[369]|8[0235689]|9[124579])|2(?:13|3[34679])|3(?:26|"    .
                  "3[69]|4[69]|5[25689]|6[124579])|4(?:4[4578]|6[0369]|"     .
                  "7[38]|8[258]|9[0134679])|5(?:0[0235689]|1[124579])|"      .
                  "6(?:15|32|43|5[29]|66|7[169]|8[0235689]|9[124578])|"      .
                  "70[0134679])|"                                            .
              "6(?:0(?:4[79]|5[02])|1(?:0[36]|1[047]|2[0369]|3[258]|"        .
                  "4[25689]|5[124578]|6[0134679]|7[0235689]|8[124578]|"      .
                  "9[0134679])|2(?:15|24|3[17]|4[27]|5[037]|6[0489]|"        .
                  "7[124579])|3(?:17|28|3[27]|4[269]|5[258]|6[14579])|"      .
                  "4(?:50|65|7[269]|8[24679])|5(?:15|2[348]))|"              .
              "7(?:0(?:7[02468]|8[024])|199|2(?:0[49]|1[58]|2[258]|"         .
                  "3[024679]|4[134679]|5[0235689]|6[124578]|7[0134679]|"     .
                  "8[0235689]|9[124579])|3(?:18|20|3[247]|4[0268]|"          .
                  "5[03579])|4(?:12|2[124]|37|4[07]|5[036]|6[149]|7[58]|"    .
                  "8[368]|9[0134679])|5(?:0[0235689]|1[134679]|"             .
                  "2[0235689]|3[124579])|6(?:1[68]|3[138]|4[057]|"           .
                  "5[034679]|88)|7(?:0[258]|1[147]|2[03-579]|37|53|6[29]|"   .
                  "7[235689]|8[0235689]|9[124579])|8(?:16|28|3[34679]|"      .
                  "4[0235689]|5[124579]|77|9[26])|9(?:0[0134679]|22|"        .
                  "4[147]|5[03679]|80|9[0369]))|"                            .
              "8(?:5(?:2[7-9]|30|4[47]|5[349]|74|87|9[03679])|6(?:17|"       .
                  "3[149]|46|6[03679]|7[38]|93)|7(?:0[148]|1[146]|24|39|"    .
                  "4[3469]))|"                                               .
              "9(?:0(?:8[4-79]|9[1246-9])|1(?:0[02]|30|89|9[258])|3(?:10|"   .
                  "26|3[048])|4(?:2[3578]|3[89]|4[148])|51[08]|6(?:10|"      .
                  "2[58]|3[1468])|7(?:06|1[38]|3[45]|5[259]|6[258])|"        .
                  "8(?:1[79]|26|3[0147]|4[268]|6[79]|8[057]|9[1478])|"       .
                  "9(?:47|5[58]|7[46]|8[68]|9[1468])))"                      .
    ")",


    #
    # Postal codes of Greenland use a slice of the Danish postal code system.
    # Greenlands postal code consist of 4 digits, the first two being 39.
    # Except Santas postal code. He uses 2412.
    #
    # Data from: http://download.geonames.org/export/zip/GL.zip
    #
    Greenland   =>
      "(?k:2412"                                                             .
         "|39(?:0[05]|1[0-359]|2[0-4]|3[02]|40|5[0-35]|6[124]|"              .
               "7[0-2]|8[0245]|92)"                                          .
    ")",


    #
    # Postal codes for Italy use 5 digits, with leading 0s. Codes starting
    # with 4789 belong to San Marino.
    #
    # Data from: http://download.geonames.org/export/zip/IT.zip
    # 

    Italy       =>
      "(?k:0(?:0(?:0(?:1[0-357-9]|2[0-9]|3[0-9]|4[0-9]|5[0-57-9]|"           .
                  "6[0-35-9])|1(?:19|2[0-8]|3[1-9]|4[1-9]|5[1-9]|6[1-9]|"    .
                  "7[1-9]|8[1-9]|9[1-9]))|"                                  .
              "1(?:0(?:1[0-24-9]|2[0-8]|3[02-9])|100)|"                      .
              "2(?:0(?:1[0-689]|2[0-68]|3[0-57-9]|4[0-9])|100)|"             .
              "3(?:0(?:1[0-46-9]|2[0-9]|3[0-9]|4[0-9])|100)|"                .
              "4(?:0(?:1[0-9]|2[0-9])|100)|"                                 .
              "5(?:0(?:1[0-8]|2[0-689]|3[0-2459])|100)|"                     .
              "6(?:0(?:1[024689]|2[0-9]|3[013-689]|4[0-79]|5[013-79]|"       .
                  "6[0-689]|7[023]|8[1349])|1(?:00|2[1-9]|3[124]))|"         .
              "7(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[013-9])|100)|"                 .
              "8(?:0(?:1[0-35-9]|2[0-9]|3[0-9]|4[02-9])|100)|"               .
              "9(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-57-9]|7[0-9]|8[0-68]|"       .
                  "9[0-9])|1(?:00|2[1-9]|3[14]|70)))"                        .

         "|1(?:0(?:0(?:1[0-9]|2[02-689]|3[0-24-8]|4[0-68]|5[0-9]|6[0-9]|"    .
                  "7[0-8]|8[0-8]|9[0-589])|1(?:00|2[1-9]|3[1-9]|4[1-9]|"     .
                  "5[1-6]))|"                                                .
              "1(?:0(?:1[0-8]|2[0-9])|100)|"                                 .
              "2(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-68]|6[0-689]|"        .
                  "7[0-9]|8[0-47-9])|100)|"                                  .
              "3(?:0(?:1[0-27-9]|2[0-8]|3[0-9]|4[013-9]|60)|100|"            .
                  "8(?:1[124-8]|2[1-5]|3[13-6]|4[13-578]|5[1-6]|6[1-8]|"     .
                  "7[1-8]|8[1-8]|9[13-9])|900)|"                             .
              "4(?:0(?:1[0-9]|2[0-6]|3[0-79]|4[0-9]|5[0-57-9])|100)|"        .
              "5(?:0(?:1[0-9]|2[0-9]|3[0-689]|4[0-689]|5[0-9]|6[0-9]|"       .
                  "7[0-9])|100)|"                                            .
              "6(?:0(?:1[0-9]|2[0-9]|3[0-689]|4[0-9])|1(?:00|2[1-9]|"        .
                  "3[1-9]|4[1-9]|5[1-9]|6[1-7]))|"                           .
              "7(?:0(?:1[0-579]|2[0-8]|3[0-57-9]|4[0-8]|5[1-8])|100)|"       .
              "8(?:0(?:1[0-9]|2[0-7]|3[0-9])|100)|"                          .
              "9(?:0(?:1[0-8]|2[0158]|3[0-478])|1(?:00|2[1-6]|3[1-9])))"     .

         "|2(?:0(?:0(?:1[0-9]|2[0-9]|3[027]|40|56|6[0-9]|7[078]|8[0-9]|"     .
                  "9[0-9])|1(?:2[1-9]|3[1-9]|4[1-9]|5[1-9]|6[12])|"          .
                  "8(?:1[1-6]|2[1-6]|3[1-8]|4[1-7]|5[1-7]|6[1-7]|7[1-7]|"    .
                  "8[1-6])|900)|"                                            .
              "1(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9])|100)|"            .
              "2(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[04-6]|6[0369]|7[0-9])|100)|"   .
              "3(?:0(?:1[0-9]|2[0-79]|3[0-8])|100|8(?:0[124-8]|1[13-9]|"     .
                  "2[1-9]|3[1-8]|4[1-9]|5[12457]|6[124578]|7[013-9]|"        .
                  "8[013-9]|9[0-9])|900)|"                                   .
              "4(?:0(?:1[0-9]|2[0-9]|3[013-9]|4[0-9]|5[0-9]|6[0-9])|"        .
                  "1(?:00|2[1-9]))|"                                         .
              "5(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-589]|"         .
                  "7[0-9]|8[0-9])|1(?:00|2[1-9]|3[1-6]))|"                   .
              "6(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9])|100|8(?:1[1-8]|"         .
                  "2[1-8]|3[1-9]|4[1-9]|5[1-9]|6[1-7])|900)|"                .
              "7(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-57-9])|100)|"         .
              "8(?:0(?:1[0-79]|2[148]|4[0135-7]|5[03]|6[0-24-689]|"          .
                  "7[0-9])|100|8(?:0[1-5]|1[1-9]|2[1-8]|3[1-368]|4[1-5]|"    .
                  "5[1-9]|6[1-68]|7[135-79]|8[13-7]|9[13-9])|922)|"          .
              "9(?:0(?:1[0-9]|2[0-9])|100))"                                 .

         "|3(?:0(?:0(?:1[03-6]|2[0-9]|3[0-9])|1(?:00|2[1-6]|3[1-35]|"        .
                  "4[12]|7[0-5]))|"                                          .
              "1(?:0(?:1[0-8]|2[0-9]|3[0-9]|4[0-9]|5[0-9])|100)|"            .
              "2(?:0(?:1[02-6]|2[0-26-8]|3[0-7]|4[0-7])|100)|"               .
              "3(?:0(?:1[0135-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|7[024-9]|"      .
                  "8[0-7]|9[02457-9])|1(?:00|70))|"                          .
              "4(?:0(?:1[0-24-8]|7[0-9])|1(?:00|2[1-9]|3[1-9]|4[1-9]|51|"    .
                  "70))|"                                                    .
              "5(?:0(?:1[0-9]|2[0-9]|3[0-24-8]|4[0-8])|1(?:00|2[1-9]|"       .
                  "3[1-9]|4[1-3]))|"                                         .
              "6(?:0(?:1[0-6]|2[0-8]|3[0-6]|4[0235-7]|5[0-7]|6[0-6]|"        .
                  "7[0-35-8])|100)|"                                         .
              "7(?:0(?:1[0-9]|2[0-4689]|3[0-25689]|4[0-79]|5[0-9]|"          .
                  "6[02-46-9])|1(?:00|2[1-9]|3[1-9]|42))|"                   .
              "8(?:0(?:1[0-35-9]|2[0-9]|3[0-9]|4[0-35-9]|5[0-79]|6[0-9]|"    .
                  "7[0-9]|8[0-35-9])|100)|"                                  .
              "9(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-8])|100))"            .

         "|4(?:0(?:0(?:1[0-9]|2[0-7]|3[02-8]|4[1-3568]|5[0-79]|6[0-9])|"     .
                  "1(?:00|2[1-9]|3[1-9]|41))|"                               .
              "1(?:0(?:1[0-9]|2[0-35-9]|3[0-9]|4[02-689]|5[1-9])|100)|"      .
              "2(?:0(?:1[0-9]|2[0-8]|3[0-579]|4[0-9])|100)|"                 .
              "3(?:0(?:1[0-57-9]|2[0-24589]|3[025-9]|4[0-57-9]|"             .
                  "5[0-35689])|1(?:00|26))|"                                 .
              "4(?:0(?:1[0-24-69]|2[0-35-9]|3[03-579]|4[1-357-9])|100)|"     .
              "5(?:0(?:1[0-2457-9]|2[0-7]|3[0-9])|100)|"                     .
              "6(?:0(?:1[0-47-9]|2[02-9]|3[0-35-79]|4[0-9])|100)|"           .
              "7(?:0(?:1[0-9]|2[013578]|3[02459]|4[23])|1(?:00|2[12])|"      .
                  "8(?:14|2[24-6]|3[2-8]|4[1-3]|5[3-5]|6[1-7])|900)|"        .
              "8(?:0(?:1[0-578]|2[024-7])|100))"                             .

         "|5(?:0(?:0(?:1[02-489]|2[0-35-8]|3[1-9]|41|5[0-689]|6[0-8])|"      .
                  "1(?:00|2[1-79]|3[1-79]|4[1-5]))|"                         .
              "1(?:0(?:1[0-35-9]|2[0148]|3[014-79])|100)|"                   .
              "2(?:0(?:1[014-8]|2[0-24-9]|3[0-35-8]|4[13-8])|100)|"          .
              "3(?:0(?:1[1-9]|2[0-7]|3[014-7]|4[0-357-9])|100)|"             .
              "4(?:0(?:1[0-6]|2[136-9]|3[358])|100)|"                        .
              "5(?:0(?:1[0-689]|2[02357]|3[0-68]|4[0-2579]|5[14]|"           .
                  "6[0-24])|100)|"                                           .
              "6(?:0(?:1[0-279]|2[0-589]|3[0-8]|4[013-68])|1(?:00|"          .
                  "2[1-8]))|"                                                .
              "7(?:0(?:1[467]|2[0-3578]|3[0-46-9])|1(?:00|2[1-8]))|"         .
              "8(?:0(?:1[0-2457-9]|2[02-7]|3[1346-8]|4[02-5]|5[13-5])|"      .
                  "100)|"                                                    .
              "9(?:0(?:1[1356]|2[14-6])|100))"                               .

         "|6(?:0(?:0(?:1[013589]|2[0-24-7]|3[013-9]|4[01348])|1(?:00|"       .
                  "2[1-9]|31))|"                                             .
              "1(?:0(?:1[0-4]|2[0-689]|3[02-47-9]|4[0-9])|100)|"             .
              "2(?:0(?:1[0-2457-9]|2[0-24-9]|3[24-689])|100)|"               .
              "3(?:0(?:20|31|6[1-9]|7[1-9]|8[1-8]|9[1-6])|100|8(?:1[1-6]|"   .
                  "2[1-8]|3[1-9]|4[1-8]|5[1-8])|900)|"                       .
              "4(?:0(?:1[0-68]|2[013-8]|3[0-79]|4[0-79])|100)|"              .
              "5(?:0(?:1[0-579]|2[02-9])|1(?:00|2[1-9]|3[12]))|"             .
              "6(?:0(?:1[0-24-9]|2[0-36]|3[0-46-8]|4[0-7]|5[0-24])|100)|"    .
              "7(?:0(?:1[02-579]|2[0-9]|3[0-9]|4[013-9]|5[0-9]|"             .
                  "6[0-46-9])|100))"                                         .

         "|7(?:0(?:0(?:1[013-9]|2[0-9]|3[2378]|4[2-4]|5[46])|1(?:00|"        .
                  "2[1-9]|31))|"                                             .
              "1(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-3578])|100)|"                .
              "2(?:0(?:1[02-9]|2[0-9])|100)|"                                .
              "3(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9])|100)|"            .
              "4(?:0(?:1[0-9]|2[0-8])|100)|"                                 .
              "5(?:0(?:1[0-9]|2[0-9])|100)|"                                 .
              "6(?:01[1-7]|12[135]))"                                        .

         "|8(?:0(?:0(?:1[0-46-9]|2[0-9]|3[0-689]|4[0-24-9]|5[013-9]|"        .
                  "6[0-35-79]|7[013-9])|1(?:00|2[1-9]|3[1-9]|4[1-7]))|"      .
              "1(?:0(?:1[0-467]|2[0-578]|3[0-9]|4[0-4679]|5[0-9])|100)|"     .
              "2(?:0(?:1[0135-9]|2[0-9]|3[0-46-8])|100)|"                    .
              "3(?:0(?:1[0-8]|2[0-9]|3[0-24-9]|4[0-9]|5[0-9])|100)|"         .
              "4(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[02-9]|5[0-35-79]|6[0-25-9]|"   .
                  "7[03-9]|8[0-8]|9[0-25689])|1(?:00|2[1-9]|3[1-5]))|"       .
              "5(?:0(?:1[0-8]|2[0-9]|3[0-9]|4[02-46-9]|5[0-9])|100)|"        .
              "6(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|7[0-57-9]|8[0-9]|"        .
                  "9[0-7])|1(?:00|70))|"                                     .
              "7(?:0(?:1[0-9]|2[0-46-9]|3[0-8]|4[0-8]|5[0-8]|6[0-24679]|"    .
                  "7[0-6])|100)|"                                            .
              "8(?:0(?:2[0-245]|4[0-24-79]|5[014-6]|6[02457-9]|70)|100|"     .
                  "8(?:1[1-9]|2[1-5]|3[1-8]|4[12])|900)|"                    .
              "9(?:0(?:1[0-8]|2[0-9]|3[0-9]|4[0-9]|5[0246-8]|6[02-59])|"     .
                  "1(?:00|2[1-9]|3[1-5])|8(?:1[2-9]|2[1-4]|3[1-4]|4[1-4]|"   .
                  "5[1-3]|6[1-46-8])|900))"                                  .

         "|9(?:0(?:0(?:1[0-9]|2[0-9]|3[0-9]|4[0-9])|1(?:00|2[1-9]|3[1-9]|"   .
                  "4[1-9]|51))|"                                             .
              "1(?:0(?:1[0-9]|2[0-9])|100)|"                                 .
              "2(?:0(?:1[0-9]|2[0-9])|100)|"                                 .
              "3(?:01[0-9]|100)|"                                            .
              "4(?:01[0-9]|100)|"                                            .
              "5(?:0(?:1[0-9]|2[0-2457-9]|3[0-9]|4[0-9])|1(?:00|2[1-9]|"     .
                  "31))|"                                                    .
              "6(?:01[0-9]|100)|"                                            .
              "7(?:01[0-9]|100)|"                                            .
              "8(?:0(?:2[0-35-9]|3[0-9]|4[0-9]|5[013-9]|6[0-9]|7[0-9])|"     .
                  "1(?:00|2[1-9]|3[1-9]|4[1-9]|5[1-9]|6[1-8])))"             .
    ")",


    #
    # The numbering system for postal codes of Liechtenstein is part of
    # the numbering system for Swiss postal codes. Four digits are used
    # and all postal codes in Liechtenstein start with 94, the third
    # digit an 8 or a 9.
    #
    # Data from: http://download.geonames.org/export/zip/LI.zip
    #
    # The file above does not include 9489 (instead, a different file
    # reports it to be in CH), but 9489 is the postal code for Schaan Log,
    # which is located in Liechtenstein.
    #
    # http://postal-codes.findthedata.com/l/57083/9489-Schaan-Log
    #
    Liechtenstein => "(?k:94(?:8[5-9]|9[0-8]))",


    #
    # https://www.post.lu/documents/10181/2314856/EptSggaCsv.csv/
    #           ee8fa0de-5e84-4e31-8cbd-b7b57679c21a?param=0.19755403072045696
    # (http://bit.ly/1PeTVqY)
    #
    Luxembourg =>
      "(?k:0(?:1(?:01|2[13-5]|3[12]|4[1-3]|61|7[13]|8[1-3]|9[12])|"          .
              "2(?:02|11|31|41|51|6[23]|91)|"                                .
              "3(?:2[1-5]|32|42|6[12]|72|8[1-3]|91)|"                        .
              "4(?:0[1-3]|1[12]|2[12]|4[1-356]|5[13]|6[2-4]|7[12]|8[12]|"    .
                  "9[1-7])|"                                                 .
              "5(?:21|3[12]|5[1-4]|6[12]|8[1-4])|"                           .
              "6(?:1[12]|4[1-4]|6[1-4]|7[12]|9[12])|"                        .
              "7(?:2[1-3]|5[1-4]|6[13]|7[12])|"                              .
              "8(?:0[1-9]|1[1-4]|2[1-3]|3[1-4]|41|5[2-5]|61|81)|"            .
              "9(?:0[134]|1[1-4]|2[12]|4[2-4]|5[1-4]|6[1-3]|7[1-3]|"         .
                  "8[1-3]|9[1-5]))"                                          .

         "|1(?:0(?:09|1[0-9]|2[0-9]|30|50|60|90)|"                           .
              "1(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-39]|6[01])|"              .
              "2(?:0[89]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|"         .
                  "7[0-9]|8[0-3])|"                                          .
              "3(?:09|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|7[0-6])|"    .
              "4(?:09|1[1-9]|2[0-9]|3[0-4]|4[589]|5[0-9]|6[0-9]|7[0-9]|"     .
                  "8[0-4]|9[09])|"                                           .
              "5(?:0[89]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-2])|"               .
              "6(?:1[013-9]|2[0-9]|3[0-9]|4[0-9]|5[0-6]|6[01]|70)|"          .
              "7(?:09|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-3])|"                  .
              "8(?:1[1-8]|2[0-2]|3[1-9]|4[0-356]|5[0-9]|6[0-8]|7[0-3]|"      .
                  "8[0-2]|9[6-9])|"                                          .
              "9(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-4]))"                     .

         "|2(?:0(?:1[0-9]|20|8[024-9]|9[013-9])|"                            .
              "1(?:1[1-9]|2[0-57-9]|3[0-57-9]|4[0-9]|5[0-9]|6[0-9]|"         .
                  "7[0-9]|8[0-4])|"                                          .
              "2(?:1[0-5]|2[0-9]|3[0-4]|4[0-3]|6[1-35-9]|7[0-3])|"           .
              "3(?:0[89]|1[0-9]|2[02-46-9]|3[0-9]|4[0-9]|5[0-9]|6[01]|70|"   .
                  "8[01])|"                                                  .
              "4(?:09|1[0-9]|2[02-9]|3[0-6]|4[0-9]|5[0-4])|"                 .
              "5(?:1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-7])|"              .
              "6(?:09|1[0-79]|2[0-9]|3[0-7]|5[1-4]|6[1-9]|7[0-4]|8[01])|"    .
              "7(?:1[1-9]|2[0-9]|3[0-46-9]|4[01]|6[1-4])|"                   .
              "8(?:50|60|8[07-9])|"                                          .
              "9(?:00|1[0-689]|2[0-25689]|3[0-9]|4[0-269]|5[0-8]|"           .
                  "6[014-68]|7[14-9]|8[0-8]|9[0-24-9]))"                     .

         "|3(?:2(?:0[15689]|1[0-9]|2[0-59]|3[0-9]|4[0-39]|5[0-9]|"           .
                  "6[015-79]|7[0-9]|8[0-8]|90)|"                             .
              "3(?:1[13-8]|2[0-9]|3[02-7]|4[015-9]|5[0-9]|6[0-9]|7[0-8]|"    .
                  "8[02-69]|9[0-8])|"                                        .
              "4(?:0[1-39]|1[0-9]|2[0-9]|3[0-3569]|4[0-9]|5[0-4]|6[0-9]|"    .
                  "7[0-6]|8[0-9]|9[0-3])|"                                   .
              "5(?:0[1-9]|1[0-7]|2[0-69]|3[0-29]|4[0-9]|5[0-5]|6[0-9]|"      .
                  "7[0-6]|8[2-589]|9[0-8])|"                                 .
              "6(?:01|1[1-7]|2[0-2]|3[015-8]|4[0-4]|5[0-8]|60|7[0-9]|"       .
                  "8[0-2])|"                                                 .
              "7(?:0[15]|1[0-9]|2[0-9]|3[013-9]|4[0-4]|5[0-5]|6[1-8]|"       .
                  "7[0-6]|8[0-246-9]|90)|"                                   .
              "8(?:01|1[0-9]|2[0-6]|3[0-9]|4[0-4]|5[0-9]|6[0-4]|7[0-9]|"     .
                  "8[0-4]|9[5-9])|"                                          .
              "9(?:0[19]|1[0-9]|2[0-9]|3[0-9]|4[013-5]|6[01]|8[05]))"        .

         "|4(?:0(?:0[1-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|"        .
                  "7[0-4]|8[0-68])|"                                         .
              "1(?:0[0-35-9]|1[0-689]|2[0-356]|3[0-9]|4[0-39]|5[0-6]|"       .
                  "6[4-9]|7[0-9]|80)|"                                       .
              "2(?:0[1-9]|1[013467]|2[0-2]|3[0-9]|4[0-9]|5[0-6]|6[0-8]|"     .
                  "7[0-25-9]|8[0-39])|"                                      .
              "3(?:0[1-9]|1[04-9]|2[0-8]|3[0-7]|4[0-9]|5[0-5]|6[0-9]|"       .
                  "7[0-3]|8[0-5]|9[0-8])|"                                   .
              "4(?:0[125-9]|1[0-9]|2[0-39]|3[0-9]|4[0-9]|5[019]|6[0-9]|"     .
                  "7[05-9]|8[0-9]|9[0-49])|"                                 .
              "5(?:0[1-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-59]|"       .
                  "7[0-9]|8[0-3]|9[0-9])|"                                   .
              "6(?:0[1-9]|1[0-3]|2[0-9]|3[0-689]|4[0-9]|5[0136-8]|6[0-9]|"   .
                  "7[0-26-9]|8[0-9]|9[0-36-8])|"                             .
              "7(?:0[126-9]|1[0-9]|2[0-2]|3[0-9]|4[0-5]|5[0-9]|6[0-3]|"      .
                  "7[0-9]|8[0-6]|9[5-8])|"                                   .
              "8(?:0[124-9]|1[0-589]|2[02-69]|3[0-47-9]|4[02-9]|5[03]|"      .
                  "7[0-9]|8[0-9]|9[0-5])|"                                   .
              "9(?:0[126-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-689]|6[0-9]|"    .
                  "7[0-9]|8[0-24-9]|9[0-9]))"                                .

         "|5(?:2(?:01|1[1-9]|2[0-2]|3[0-9]|4[0134]|5[0-6]|80|9[09])|"        .
              "3(?:1[024-8]|2[02468]|3[0-79]|4[0-2]|5[1-359]|6[0-9]|"        .
                  "7[0-8]|80)|"                                              .
              "4(?:0[1-578]|1[0-9]|2[1-79]|3[0-59]|4[0-7]|5[01]|6[056]|"     .
                  "7[01]|8[0-3589]|9[59])|"                                  .
              "5(?:0[15-7]|1[1-9]|2[0-3]|3[0-9]|4[0489]|5[0-9]|6[01]|"       .
                  "7[0-7])|"                                                 .
              "6(?:0[125]|1[0-9]|2[06-9]|3[0-9]|40|5[0-9]|7[015]|8[0-7]|"    .
                  "9[0-358])|"                                               .
              "7(?:01|1[0-46-9]|2[0-3]|30|4[01]|5[0-6]|6[0-2]|7[0-6])|"      .
              "8(?:0[189]|1[0-9]|2[0-689]|3[0-9]|4[0-46]|5[02-69]|6[0-9]|"   .
                  "7[0-68]|8[04-9]|9[0-389])|"                               .
              "9(?:4[0-3]|5[0-9]|6[0-49]|7[0-8]))"                           .

         "|6(?:1(?:01|1[1-9]|2[0-59]|3[0-9]|4[0-9]|5[05]|6[0-3569]|"         .
                  "7[015]|8[0-9]|9[05-7])|"                                  .
              "2(?:01|1[0-5]|25|3[01589]|4[0356]|5[0-25])|"                  .
              "3(?:01|1[0-5]|40|50|60|70|80)|"                               .
              "4(?:0[12689]|1[0-9]|2[0-3]|3[0-9]|4[0-35-9]|5[0-5]|6[0-9]|"   .
                  "7[0-9]|8[0-8]|9[0-79])|"                                  .
              "5(?:5[0-357-9]|6[02]|7[0-29]|8[0-35-7]|90)|"                  .
              "6(?:01|1[0-9]|2[0-3]|3[0-8]|4[5-9]|5[01]|6[013569]|7[0-8]|"   .
                  "8[0-9]|9[0-35])|"                                         .
              "7(?:01|1[1-9]|2[0-6]|3[0-9]|4[0-5]|5[0-9]|6[0-5]|7[0-9]|"     .
                  "8[0-6]|9[0-6])|"                                          .
              "8(?:15|3[0-9]|4[0-2]|5[0-28]|6[89]|7[01]|80)|"                .
              "9(?:0[156]|1[0-9]|2[1-356]|3[01349]|4[0-8]|5[0156]|"          .
                  "6[0-29]|7[0-9]|8[0-25-9]|9[0-25-9]))"                     .

         "|7(?:2(?:0[129]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-489]|"      .
                  "7[04])|"                                                  .
              "3(?:0[2-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-4]|"        .
                  "7[0-9]|8[0-4]|9[0-25-7])|"                                .
              "4(?:09|1[0-25-8]|2[013-5]|3[0-35]|4[0-9]|5[0-8]|6[0-5]|"      .
                  "7[0135]|8[0-2])|"                                         .
              "5(?:0[125-8]|1[1-9]|2[0-6]|3[1-9]|4[0-7]|5[3-9]|6[0-9]|"      .
                  "7[0-2]|9[0-9])|"                                          .
              "6(?:01|1[023589]|2[0-7]|3[3-69]|4[019]|5[0-3]|6[0-4]|"        .
                  "7[03]|8[01])|"                                            .
              "7(?:0[13]|1[0-6]|2[0-467]|3[0-57-9]|4[01]|5[089]|6[0-9]|"     .
                  "7[0-47-9]|8[0-9]|9[0-5]))"                                .

         "|8(?:0(?:0[125-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]|6[0-9]|"      .
                  "7[0-9]|8[0-9]|9[0-9])|"                                   .
              "1(?:1[0-25-9]|2[013-9]|3[0-57-9]|4[0-25-79]|5[0-46-9]|"       .
                  "6[05-7]|79|8[0-9]|9[01])|"                                .
              "2(?:0[159]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-589]|6[0-9]|"      .
                  "7[0-47-9]|8[0-9]|9[0-5])|"                                .
              "3(?:0[1-3689]|1[0-9]|2[0-9]|3[0-9]|4[0-5]|5[0-9]|6[0-9]|"     .
                  "7[0-689]|8[013-8]|9[0-689])|"                             .
              "4(?:01|1[0-5]|2[0-4]|3[5-8]|4[0-57-9]|5[0-3]|6[0-9]|"         .
                  "7[0-9]|80)|"                                              .
              "5(?:0[126-9]|1[01]|2[1-35-9]|3[0137-9]|4[0-6]|5[0-278]|"      .
                  "6[0-2])|"                                                 .
              "6(?:0[69]|1[0-5]|20)|"                                        .
              "7(?:0[15-8]|1[015]|20)|"                                      .
              "8(?:0[125689]|1[1-46-9]|2[0-6]|3[1-58]))"                     .

         "|9(?:0(?:0[1-46]|1[0-9]|2[0-9]|3[0-3]|4[0-8]|5[0-6]|6[0-9]|"       .
                  "7[0-2]|8[0-9]|9[0-489])|"                                 .
              "1(?:1[5-9]|2[0-9]|3[0-7]|4[024-7]|5[013-8]|6[013-9]|"         .
                  "7[0-35-79]|8[0-4689]|9[01])|"                             .
              "2(?:0[1-9]|1[0-8]|2[014-9]|3[0-9]|4[0-59]|5[0-9]|6[0-8]|"     .
                  "7[35-9]|8[0-9]|9[0-4])|"                                  .
              "3(?:30|40|5[0-79]|6[014-689]|7[0-35-8]|8[0-2]|9[0-25])|"      .
              "4(?:0[15-9]|1[0-24-9]|2[0-6]|40|5[1-9]|6[1-6])|"              .
              "5(?:0[126]|1[0-9]|2[0-3]|3[0-9]|4[0-5]|5[0-9]|60|7[0-9])|"    .
              "6(?:3[1-35-9]|4[013-578]|5[013-79]|6[0235689]|7[0-468]|"      .
                  "8[12479]|9[06])|"                                         .
              "7(?:0[16-9]|1[0-5]|3[78]|4[0-9]|5[1-9]|6[0-9]|7[0-69]|80)|"   .
              "8(?:0[15-9]|3[013-9]|4[01])|"                                 .
              "9(?:0[1-35-9]|1[0-3]|4[02-468]|5[0246]|6[02468]|7[0246]|"     .
                  "8[02]|9[0-39]))"                                          .
    ")",


    #
    # Postal codes of Monaco are part of the system used for France.
    # Monaco uses postal codes starting with 980. 98000 is for all
    # physical addresses, while numbers ending in 01 - 99 are reserved
    # for special deliveries.
    #
    # http://www.upu.int/fileadmin/documentsFiles/activities/
    #                                                addressingUnit/mcoEn.pdf
    #
    Monaco => "(?k:980[0-9][0-9])",

    #
    # Postal codes in Norway use 4 digits. Leading 0s happen, but not all
    # combinations are used.
    #
    # Data from: http://download.geonames.org/export/zip/GL.zip
    #
    Norway      =>
      "(?k:0(?:0(?:01|1[058]|2[14-68]|3[0-47]|4[05-8]|5[015]|60|80)|"        .
              "1(?:0[1-79]|1[0-9]|2[0-589]|3[0-9]|5[0-57-9]|6[0-24-9]|"      .
                  "7[0-9]|8[0-8]|9[0-68])|"                                  .
              "2(?:0[1-478]|1[1-8]|30|4[047]|5[0-9]|6[02-8]|7[0-9]|"         .
                  "8[0-467])|"                                               .
              "3(?:0[1-9]|1[13-9]|23|30|4[09]|5[0-9]|6[0-9]|7[0-9]|"         .
                  "8[0-3])|"                                                 .
              "4(?:0[1-689]|1[0-35]|2[1-4]|4[0-25]|5[0-24-9]|6[0-57-9]|"     .
                  "7[02-9]|8[0-9]|9[0-6])|"                                  .
              "5(?:0[1-9]|1[0-35-8]|20|40|5[0-9]|6[0-9]|7[0-9]|8[0-9]|"      .
                  "9[0-8])|"                                                 .
              "6(?:0[1-9]|1[1-9]|2[0-46]|5[0-9]|6[0-9]|7[0-9]|8[0-9]|"       .
                  "9[0-4])|"                                                 .
              "7(?:0[125]|1[02]|5[0-8]|6[03-8]|7[0-9]|8[1-9]|9[01])|"        .
              "8(?:0[15-7]|40|5[0-8]|6[0-4]|7[0-7]|8[0-4]|9[01])|"           .
              "9(?:0[1-578]|1[3-5]|5[0-9]|6[02-489]|7[0-35-9]|8[0-8]))"      .

         "|1(?:0(?:0[135-9]|11|5[1-6]|6[1-57-9]|71|8[1346-9])|"              .
              "1(?:0[129]|12|5[0-8]|6[0-9]|7[026-9]|8[12457-9])|"            .
              "2(?:0[1-57]|1[45]|5[0-9]|6[236]|7[0-589]|8[13-6]|9[0145])|"   .
              "3(?:0[0-79]|1[1-46-9]|2[1-9]|3[0-9]|4[0-24689]|5[0-46-9]|"    .
                  "6[0-9]|7[1-35-9]|8[013-9]|9[0-79])|"                      .
              "4(?:0[0-9]|1[0-9]|2[019]|3[0-2]|4[0-9]|5[013-578]|6[89]|"     .
                  "7[0-9]|8[0-8])|"                                          .
              "5(?:0[1-4689]|1[0-9]|2[0-689]|3[0-9]|4[015]|5[056]|60|70|"    .
                  "8[01]|9[0-36-9])|"                                        .
              "6(?:0[1-9]|1[02-9]|2[014-689]|3[02-46-9]|4[0-2]|"             .
                  "5[013-57-9]|6[1-7]|7[0-35689]|8[02-4]|9[02])|"            .
              "7(?:0[1-9]|1[0-589]|2[0-7]|3[03-589]|4[0235-7]|5[1-479]|"     .
                  "6[013-9]|7[126-9]|8[1-9]|9[0-4689])|"                     .
              "8(?:0[1-9]|1[1-6]|2[0357]|3[0-3]|5[019]|6[0167]|7[0158]|"     .
                  "80|9[0-3])|"                                              .
              "9(?:0[013]|1[0-2467]|2[013-9]|3[01]|4[01]|5[04]|6[013]|"      .
                  "7[01]))"                                                  .

         "|2(?:0(?:0[013-9]|1[0-9]|2[0-8]|3[0-6]|4[01]|5[0-8]|6[0-26-9]|"    .
                  "7[0-46]|8[01]|9[0-3])|"                                   .
              "1(?:0[01]|1[046]|2[03]|3[034]|5[01]|6[0-24-6]|70)|"           .
              "2(?:0[1-689]|1[0-46-9]|2[03-6]|3[0235]|40|56|6[014-6]|"       .
                  "7[01]|8[03])|"                                            .
              "3(?:0[1-9]|1[25-9]|2[0-6]|3[024-8]|4[0145]|5[035]|6[045]|"    .
                  "72|8[0-9]|9[01])|"                                        .
              "4(?:0[1-35-9]|1[0-2568]|2[0-357-9]|3[025-8]|4[0368]|5[01]|"   .
                  "60|7[6-8]|8[015])|"                                       .
              "5(?:0[01]|1[02]|4[024]|5[025]|6[01]|8[0-24])|"                .
              "6(?:0[1-9]|1[0-9]|2[4-69]|3[0-79]|4[0235-9]|5[1-36-9]|"       .
                  "6[0-9]|7[02-7]|8[02-8]|9[03-5])|"                         .
              "7(?:1[1-8]|20|30|4[023]|50|60|70)|"                           .
              "8(?:0[1-9]|1[015-9]|2[0-257]|3[0-26-9]|4[036-9]|5[013478]|"   .
                  "6[0-2467]|7[09]|8[0-2]|9[03])|"                           .
              "9(?:0[017]|1[078]|2[039]|3[03679]|4[03]|5[02-49]|6[067]|"     .
                  "7[3-57]|85))"                                             .

         "|3(?:0(?:0[1-9]|1[1-9]|2[1-9]|3[0-9]|4[0-8]|5[013-8]|6[01]|"       .
                  "7[015]|8[0-9]|9[025])|"                                   .
              "1(?:0[13-9]|1[0-9]|2[0-8]|3[1-357-9]|4[0-58]|5[0-47-9]|"      .
                  "6[0-9]|7[0-9]|8[0-9]|9[1-69])|"                           .
              "2(?:0[1-9]|1[0-9]|2[0-9]|3[0-9]|4[0-69]|5[1-9]|6[0-57-9]|"    .
                  "7[014-7]|8[0-245]|9[0-24-7])|"                            .
              "3(?:0[01]|2[0-2]|3[01]|4[0-2]|5[01589]|6[01]|7[01])|"         .
              "4(?:0[1-9]|1[0-4]|2[015-8]|3[01]|4[0-2]|7[0-24-8]|8[0-4]|"    .
                  "9[01])|"                                                  .
              "5(?:0[1-47]|1[0-9]|2[0-689]|3[013-9]|4[014]|5[01]|6[01]|"     .
                  "7[015-79]|8[018]|9[35])|"                                 .
              "6(?:0[1-689]|1[0-9]|2[0-46-9]|3[0-2]|4[6-8]|5[0268]|"         .
                  "6[0156]|7[1-9]|8[0134]|9[0-27])|"                         .
              "7(?:0[1-578]|1[0-9]|2[0-9]|3[0-9]|4[0-46-9]|5[03]|6[06]|"     .
                  "70|8[01357-9]|9[0-68])|"                                  .
              "8(?:0[0-5]|1[0-2]|2[05]|3[0-6]|4[01489]|5[03-5]|64|70|"       .
                  "8[02-8]|9[0135])|"                                        .
              "9(?:0[1-6]|1[0-9]|2[0-24589]|3[013679]|4[0-46-9]|50|"         .
                  "6[05-7]|70|9[13-9]))"                                     .

         "|4(?:0(?:0[1-9]|1[0-9]|2[0-9]|3[1-6]|4[1-9]|5[0-9]|6[4-9]|"        .
                  "7[06-9]|8[1-9]|9[0-9])|"                                  .
              "1(?:0[02]|1[09]|2[0346-9]|3[0479]|4[68]|5[02-4689]|"          .
                  "6[01347-9]|7[034]|8[027]|98)|"                            .
              "2(?:0[0189]|3[03-579]|4[04]|50|6[0245]|7[024-6]|80|"          .
                  "9[14-9])|"                                                .
              "3(?:0[1-9]|1[0-9]|2[1-9]|3[02359]|4[0-9]|5[2-8]|"             .
                  "6[02-57-9]|7[0-689]|8[0179]|9[1-8])|"                     .
              "4(?:0[0-3]|20|3[2468]|4[013]|6[0235]|73|8[045]|9[0-2])|"      .
              "5(?:0[1-69]|1[3-79]|2[0-589]|3[246]|4[04]|5[0-478]|6[03]|"    .
                  "7[5-79]|8[068]|9[056])|"                                  .
              "6(?:0[4-689]|1[0-9]|2[0-689]|3[0-9]|4[05-7]|5[16-9]|"         .
                  "6[1-6]|7[0-9]|8[1-9]|9[13-9])|"                           .
              "7(?:0[0-35]|15|2[04]|3[03-57]|4[12578]|5[4-6]|6[068]|70|"     .
                  "80|9[0-5])|"                                              .
              "8(?:0[1-489]|1[025-8]|2[013-578]|3[024689]|4[1-46-9]|"        .
                  "5[1-9]|6[1-589]|7[06-9]|8[4-9]|9[1-468])|"                .
              "9(?:0[0-29]|1[0256]|20|34|5[0-35]|7[1-4]|8[05]|9[034]))"      .

         "|5(?:0(?:0[3-9]|1[0-9]|2[0-2]|3[1-9]|4[1-35]|5[2-9]|6[378]|"       .
                  "7[235]|8[129]|9[346-9])|"                                 .
              "1(?:0[14-9]|1[13-9]|2[124]|3[0-24-7]|4[1-8]|5[1-5]|6[0-5]|"   .
                  "7[0-46-9]|8[34])|"                                        .
              "2(?:0[0-36-9]|1[0-25-8]|2[1-9]|3[0-25-9]|4[34]|5[1-47-9]|"    .
                  "6[0-578]|8[1-6]|9[139])|"                                 .
              "3(?:0[0-9]|1[04589]|2[1-35-79]|3[13-7]|4[1-35-7]|"            .
                  "5[03-578]|6[0356]|7[1489]|8[0-24578]|9[2-46-9])|"         .
              "4(?:0[1-46-9]|1[0-9]|2[0378]|3[07]|4[03-579]|5[0-57-9]|"      .
                  "6[02-5]|7[02-6]|8[046]|9[89])|"                           .
              "5(?:0[1-9]|1[124-9]|2[1-357-9]|3[1-8]|4[124-9]|5[01459]|"     .
                  "6[0135-8]|7[04-68]|8[02-689]|9[013-68])|"                 .
              "6(?:0[0-245]|1[024]|2[06-9]|3[025-7]|4[0-35-79]|5[023]|"      .
                  "8[0357]|9[03-6])|"                                        .
              "7(?:0[0-9]|1[0-589]|2[1-9]|3[01346]|4[1-35-9]|5[0-2]|"        .
                  "6[03]|7[036-9]|8[0-8])|"                                  .
              "8(?:0[3-9]|1[0-9]|2[0-24589]|3[1568]|4[157-9]|5[1-47-9]|"     .
                  "6[1-489]|7[1-36-9]|8[1246-9]|9[23569])|"                  .
              "9(?:0[2-46-8]|1[1-8]|3[16-9]|4[1378]|5[1-7]|6[0-267]|"        .
                  "7[07-9]|8[13467]|9[134]))"                                .

         "|6(?:0(?:0[1-9]|1[0-9]|2[0-68]|3[05-9]|4[03-8]|5[0-257-9]|"        .
                  "6[02-57-9]|7[06]|8[02-579]|9[0-24-689])|"                 .
              "1(?:0[0-6]|10|20|3[39]|4[0-469]|5[0-6]|6[0156]|7[04]|"        .
                  "8[34]|9[06])|"                                            .
              "2(?:0[01]|1[0-68]|2[024]|3[089]|4[09]|5[09]|6[03-5]|7[02]|"   .
                  "8[0-35]|9[02-4])|"                                        .
              "3(?:0[01]|1[05]|20|3[09]|50|6[034]|8[67]|9[0-9])|"            .
              "4(?:0[1-57-9]|1[0-689]|2[1-359]|3[013-6]|4[03-57]|5[03-7]|"   .
                  "6[0-2]|7[0256]|8[01346-8]|9[0349])|"                      .
              "5(?:0[1-46-9]|1[0-24-8]|2[0-59]|3[089]|7[01]|90)|"            .
              "6(?:0[01]|1[0-3]|2[0289]|3[0136-9]|4[02-5]|5[0235-9]|"        .
                  "7[04]|8[036-9]|9[0347-9])|"                               .
              "7(?:0[0-478]|1[013-9]|2[136-9]|3[0147]|4[01]|5[01]|6[13]|"    .
                  "7[0-26-9]|8[1-489]|9[1-35-9])|"                           .
              "8(?:0[0-9]|1[0-57-9]|2[1-36-9]|4[1378]|5[1-689]|6[13689]|"    .
                  "7[0-35-9]|8[124-8]|9[13-689])|"                           .
              "9(?:0[0-39]|1[24-9]|2[146-9]|4[0-2467]|5[1378]|6[1346-9]|"    .
                  "7[13578]|8[0-8]|9[1356]))"                                .

         "|7(?:0(?:0[3-6]|1[0-689]|2[0-9]|3[0-46-9]|4[0-9]|5[0-46-9]|"       .
                  "7[024589]|8[0-389]|9[127-9])|"                            .
              "1(?:0[015]|1[02-49]|2[015-79]|30|4[02]|5[02369]|6[05-9]|"     .
                  "7[06-8]|80|9[04])|"                                       .
              "2(?:0[0136]|1[1-3]|2[13478]|3[124-689]|4[0-367]|5[025-79]|"   .
                  "6[013468]|7[03]|8[024-9]|9[0158])|"                       .
              "3(?:0[0-2]|1[05689]|2[0179]|3[1-68]|4[0-35]|5[013-578]|61|"   .
                  "7[024]|8[03467]|9[1-37-9])|"                              .
              "4(?:0[0-9]|1[0-9]|2[0-24-9]|3[0-9]|4[0-9]|5[0-9]|"            .
                  "6[1-35-9]|7[0-9]|8[0-9]|9[0-7])|"                         .
              "5(?:0[0-9]|1[02-479]|2[059]|3[013]|4[019]|5[01]|6[0236]|"     .
                  "70|8[0134]|9[016])|"                                      .
              "6(?:0[0-9]|1[09]|2[02-49]|3[0-4]|5[0-8]|60|7[01]|90)|"        .
              "7(?:0[1-57-9]|1[0-8]|2[4-69]|3[024-9]|4[024-68]|5[01]|"       .
                  "6[01]|7[017]|9[0167])|"                                   .
              "8(?:0[0-58]|1[07-9]|2[02]|56|6[0349]|7[013]|8[24]|"           .
                  "9[02368])|"                                               .
              "9(?:0[01]|4[04]|50|60|7[01367]|8[0-25]|9[034]))"              .

         "|8(?:0(?:0[1-9]|1[0-69]|2[0-36-9]|3[0-278]|4[17-9]|5[068]|"        .
                  "6[34]|7[0-69]|8[46-9]|9[1-8])|"                           .
              "1(?:0[0238]|1[048]|2[08]|3[0568]|4[0569]|5[017-9]|6[018]|"    .
                  "7[08]|8[124-9]|9[035-8])|"                                .
              "2(?:0[0-35-9]|1[014589]|2[06]|3[0-3]|5[0156]|6[0146]|"        .
                  "7[013-68]|8[135689]|9[01478])|"                           .
              "3(?:0[0159]|1[0-7]|2[02-68]|40|5[27]|60|7[0236-8]|"           .
                  "8[02478]|9[0238])|"                                       .
              "4(?:0[0-9]|1[0-6]|2[68]|3[0289]|4[57]|5[059]|6[59]|7[05]|"    .
                  "8[013-589]|93)|"                                          .
              "5(?:0[1-9]|1[02-8]|2[023]|3[013-69]|4[036]|9[01])|"           .
              "6(?:0[1-478]|1[03-8]|2[246]|3[048]|4[0-36-8]|5[124-9]|"       .
                  "6[013-6]|72|8[01]|9[01])|"                                .
              "7(?:0[01]|2[03-5]|3[0235]|4[023]|5[02-4]|6[12467]|70)|"       .
              "8(?:0[0-59]|13|2[07]|30|4[24]|5[0-24]|6[015]|70|80|"          .
                  "9[0-27])|"                                                .
              "9(?:0[0-24-9]|10|2[0-2]|6[01]|76|8[015]))"                    .

         "|9(?:0(?:0[6-9]|1[0-9]|2[02479]|3[0478]|4[02369]|5[05-79]|"        .
                  "6[02489])|"                                               .
              "1(?:0[0-8]|1[089]|2[08]|3[0-24-8]|4[0-46-8]|5[1-369]|"        .
                  "6[1-39]|8[0-24-79]|9[02-57])|"                            .
              "2(?:40|5[1-9]|6[0-35-9]|7[0-9]|8[0-8]|9[0-4689])|"            .
              "3(?:0[02-6]|1[0156]|2[12569]|3[4-6]|5[0578]|6[05]|"           .
                  "7[02369]|8[0-24-9]|9[1-35])|"                             .
              "4(?:0[2-9]|1[1459]|2[03-7]|3[069]|4[0-8]|5[013-5]|"           .
                  "7[01569]|8[0-9]|9[6-8])|"                                 .
              "5(?:0[1-9]|1[0-9]|2[015]|3[1-36]|4[05]|5[01]|8[02-7]|"        .
                  "9[035])|"                                                 .
              "6(?:0[09]|1[056]|2[014]|5[07]|64|7[02]|9[0-2])|"              .
              "7(?:0[09]|1[0-7]|22|3[05]|4[02]|5[01]|6[03-58]|7[0-35]|82|"   .
                  "90)|"                                                     .
              "8(?:0[02]|1[015]|2[06]|4[056])|"                              .
              "9(?:00|1[024-7]|25|3[05]|5[01]|60|8[0-2]|9[01]))"             .
    ")",


    #
    # San Marino uses a slice of the postal codes for Italy. 
    # Any postal code starting with 4789, followed by another 
    # digit is from San Marino
    #
    # Data: http://download.geonames.org/export/zip/SM.zip
    #
   'San Marino' => "(?k:4789[0-9])",


    Spain       =>  "(?k:(?k:0[1-9]|[1-4][0-9]|5[0-2])(?k:[0-9])(?k:[0-9]{2}))",
                    # Five digits, first two indicate the province.
                    # Third digit: large town, main delivery rounds.
                    # Last 2 digits: delivery area, secondary delivery route
                    #                or link to rural areas.

    #
    # Switzerland uses four digit postal codes; leading 0s are not used.
    # Not every combination is in use. Postal codes starting with 948 and
    # 949 are used by Liechtenstein, and will not be recognized by the
    # pattern below.
    #
    # Data from: http://download.geonames.org/export/zip/CH.zip
    #
    Switzerland =>
      "(?k:1(?:0(?:0[0-9]|1[0-2457-9]|2[02-9]|3[0-9]|4[0-7]|5[2-589]|"       .
                  "6[1-368]|7[0-36-8]|8[0-58]|9[0-8])|"                      .
              "1(?:1[02-7]|2[1-8]|3[124-6]|4[1-9]|6[2-9]|7[02-6]|8[02-9]|"   .
                  "9[5-7])|"                                                 .
              "2(?:0[0-9]|1[1-9]|2[02-8]|3[1-4679]|4[0-8]|5[1-8]|6[0-9]|"    .
                  "7[0-9]|8[13-9]|9[0-9])|"                                  .
              "3(?:0[02-8]|1[0-35-8]|2[0-69]|3[078]|4[1-8]|5[02-8]|"         .
                  "7[2-7])|"                                                 .
              "4(?:0[014-9]|1[0235-8]|2[0-9]|3[0-9]|4[0-356]|5[02-4]|"       .
                  "6[2-48]|7[03-5]|8[2-69])|"                                .
              "5(?:09|1[02-5]|2[1-9]|3[02-8]|4[1-5]|5[1-5]|6[2-8]|"          .
                  "8[02-9]|95)|"                                             .
              "6(?:0[7-9]|1[0-9]|2[3-8]|3[0-8]|4[2-9]|5[1-46-9]|"            .
                  "6[0135-79]|7[03-9]|8[0-9]|9[0-24-79])|"                   .
              "7(?:0[0-2457-9]|1[2-9]|2[0-8]|3[0-8]|4[0-24-9]|5[2-467]|"     .
                  "6[23]|7[2-6]|8[2-9]|9[1-7])|"                             .
              "8(?:0[0-9]|1[14-8]|2[02-4]|3[23]|4[4-7]|5[2-46]|6[02-9]|"     .
                  "7[0-5]|8[0245]|9[0-35-9])|"                               .
              "9(?:0[2-8]|1[1-489]|2[0-35-9]|3[2-46-8]|4[1-8]|5[013578]|"    .
                  "6[1-9]|7[1-8]|8[1-8]|9[1-467]))"                          .

         "|2(?:0(?:0[0-46-9]|1[02-79]|2[2-578]|3[4-7]|4[236]|5[2-46-8]|"     .
                  "6[3578]|7[2-5]|8[78])|"                                   .
              "1(?:0[358]|1[2-7]|2[3467]|49)|"                               .
              "20[6-8]|3(?:0[0-46]|1[468]|2[25]|3[368]|4[05]|5[034]|"        .
                  "6[02-4])|"                                                .
              "4(?:0[056]|1[46])|"                                           .
              "5(?:0[0-5]|1[02-8]|2[035]|3[2-8]|4[02-5]|5[2-8]|6[02-5]|"     .
                  "7[25-7])|"                                                .
              "6(?:0[3-8]|1[02356])|"                                        .
              "7(?:1[02-8]|2[023]|3[23568]|4[02-8]|62)|"                     .
              "8(?:0[0235-7]|1[2-4]|2[2-9]|3[02]|4[23]|5[2-7]|6[34]|73|"     .
                  "8[2-9])|"                                                 .
              "9(?:0[02-8]|1[24-6]|2[2-6]|3[235]|4[2-467]|5[02-4]))"         .

         "|3(?:0(?:0[0-8]|1[0-57-9]|2[0479]|3[02-9]|4[0-9]|5[02-4]|"         .
                  "6[35-8]|7[0-8]|8[2-9]|9[5-9])|"                           .
              "1(?:1[0-6]|2[2-8]|32|4[4578]|5[02-9]|7[2-9]|8[2-6])|"         .
              "2(?:0[2-8]|1[02-6]|2[56]|3[2-8]|5[0-7]|6[2-46-8]|7[0-4]|"     .
                  "8[02-6]|9[2-8])|"                                         .
              "3(?:0[235-9]|1[2-57]|2[1-6]|6[0235-8]|7[2-7]|80)|"            .
              "4(?:0[0-2]|1[2-9]|2[1-9]|3[2-9]|5[2-7]|6[2-5]|7[2-6])|"       .
              "5(?:0[346-8]|1[023]|3[1-8]|43|5[0-35-7])|"                    .
              "6(?:0[0-57-9]|1[2-9]|2[2-9]|3[1-68]|4[5-7]|5[2-8]|6[1-5]|"    .
                  "7[1-4])|"                                                 .
              "7(?:0[02-7]|1[13-8]|2[2-5]|5[2-8]|6[2-6]|7[0-35-8]|8[0-5]|"   .
                  "92)|"                                                     .
              "8(?:0[013-7]|1[2-68]|2[2-6]|5[2-8]|6[02-4])|"                 .
              "9(?:0[0-8]|1[0-46-9]|2[02-9]|3[0-57-9]|4[02-9]|5[1-7]|"       .
                  "6[0135-8]|7[0-9]|8[2-9]|9[1-9]))"                         .

         "|4(?:0(?:0[0-57-9]|1[0-35-9]|2[03-5]|3[0-59]|4[0-2]|5[1-9]|65|"    .
                  "7[058]|8[0-9]|9[1-6])|"                                   .
              "1(?:0[1-8]|1[24-8]|2[3-7]|3[23]|4[2-8]|53)|"                  .
              "2(?:0[2-46-8]|2[2-9]|3[2-4]|4[2-7]|5[2-4])|"                  .
              "3(?:0[2-5]|1[02-7]|2[2-5]|3[2-4])|"                           .
              "4(?:02|1[0-9]|2[1-6]|3[1-8]|4[1-8]|5[0-35-8]|6[0-9]|"         .
                  "9[2-7])|"                                                 .
              "5(?:0[0-39]|1[2-5]|2[2-58]|3[2-9]|4[23]|5[1-46-8]|6[2-6]|"    .
                  "7[1346-9]|8[1-8])|"                                       .
              "6(?:0[0139]|1[1-8]|2[0-689]|3[02-4]|40|5[2-8]|6[35])|"        .
              "7(?:0[2-4]|1[02-9])|"                                         .
              "8(?:0[0-35-9]|1[2-4]|5[236])|"                                .
              "9(?:0[0-2]|1[1-79]|2[2-4]|3[2-8]|4[2-4]|5[02-5]))"            .

         "|5(?:0(?:0[014]|1[02-578]|2[2-8]|3[2-7]|4[02-46]|5[346-8]|"        .
                  "6[2-4]|7[02-9]|8[02-5])|"                                 .
              "1(?:0[235-8]|1[236])|"                                        .
              "2(?:0[01]|1[023]|2[2-5]|3[2-7]|4[2-6]|7[2-7])|"               .
              "3(?:0[013-6]|1[2-8]|2[2-6]|3[02-4])|"                         .
              "4(?:0[0-24-68]|1[235-7]|2[0356]|3[0-26]|4[2-5]|5[2-4]|"       .
                  "6[2-7])|"                                                 .
              "5(?:0[2-7]|12|2[245])|"                                       .
              "6(?:0[013-8]|1[0-9]|2[0-8]|3[02467]|4[2-7])|"                 .
              "7(?:0[2-8]|12|2[2-8]|3[2-7]|4[256]))"                         .

         "|6(?:0(?:0[02-9]|1[0-9]|2[0-8]|3[0-9]|4[2-578]|5[2356]|"           .
                  "6[0-46-8]|7[2-48]|8[3-6])|"                               .
              "1(?:0[2356]|1[02-4]|2[2356]|3[023]|4[2-7]|5[2-46]|"           .
                  "6[0-367]|7[034]|82|9[267])|"                              .
              "2(?:0[3-8]|1[0-8]|2[12]|3[1-6]|4[2-8]|5[23]|6[02-5]|"         .
                  "7[4-7]|8[013-9]|9[45])|"                                  .
              "3(?:0[0-4]|1[02-57-9]|3[0-3]|4[0-69]|5[346]|6[235]|7[0-7]|"   .
                  "8[236-8]|9[01])|"                                         .
              "4(?:0[2-5]|1[04-8]|2[2-4]|3[0-468]|4[0-3]|5[24]|6[0-9]|"      .
                  "7[2-6]|8[2457]|9[013])|"                                  .
              "5(?:0[0136]|1[1-8]|2[3-8]|3[2-578]|4[0-9]|5[6-8]|6[235]|"     .
                  "7[1-9]|8[2-4]|9[02-9])|"                                  .
              "6(?:0[0-245]|1[1-468]|22|3[1-7]|4[4-8]|5[2-9]|6[1-4]|"        .
                  "7[02-8]|8[2-5]|9[02-6])|"                                 .
              "7(?:0[2357]|1[03-9]|2[0-4]|4[2-9]|6[034]|7[2-7]|8[01])|"      .
              "8(?:0[2-9]|1[04-8]|2[1-35-8]|3[02-9]|5[02-5]|6[2-7]|"         .
                  "7[2-57]|83)|"                                             .
              "9(?:0[0-8]|1[1-9]|2[124-9]|3[02-9]|4[2-9]|5[0-9]|6[2-8]|"     .
                  "7[46-9]|8[0-46-9]|9[0-9]))"                               .

         "|7(?:0(?:0[0-467]|1[2-9]|2[36-9]|3[12]|5[06-8]|6[2-4]|7[4-8]|"     .
                  "8[2-4])|"                                                 .
              "1(?:0[4679]|1[0-6]|2[26-8]|3[02-8]|4[1-9]|5[1-9]|6[2-8]|"     .
                  "7[2-6]|8[02-9])|"                                         .
              "2(?:0[1-68]|1[2-5]|2[02-468]|3[1-35]|4[0-79]|5[02]|6[05]|"    .
                  "7[026-8])|"                                               .
              "3(?:0[2-467]|1[02-57]|2[03-6])|"                              .
              "4(?:0[2-578]|1[1-9]|2[1-8]|3[0-8]|4[02-8]|5[0-9]|6[02-4]|"    .
                  "7[237]|8[24]|9[2-4])|"                                    .
              "5(?:0[02-5]|1[2-7]|2[2-7]|3[02-7]|4[2356]|5[0-46-9]|"         .
                  "6[023])|"                                                 .
              "6(?:0[2-68]|10)|"                                             .
              "7(?:10|4[1-8]))"                                              .

         "|8(?:0(?:0[0-68]|1[0-256]|2[0-467]|3[0-46-9]|4[0-24-9]|"           .
                  "5[0-357-9]|6[013-68]|7[0145]|8[015-8]|9[0-3689])|"        .
              "1(?:0[2-9]|1[2-578]|2[1-7]|3[02-6]|4[23]|5[2-8]|6[24-6]|"     .
                  "7[2-5]|8[0-57]|9[2-7])|"                                  .
              "2(?:0[0-578]|1[02-9]|2[2-68]|3[1-689]|4[0-35-8]|5[2-59]|"     .
                  "6[0-9]|7[2-4]|8[05])|"                                    .
              "3(?:0[1-9]|1[0-2457]|2[025]|3[0-25]|4[02-5]|5[2-7]|6[023]|"   .
                  "7[0-246])|"                                               .
              "4(?:0[0-24-689]|1[0-68]|2[1-8]|4[247]|5[0-57-9]|6[0-8]|"      .
                  "7[124-9]|8[2-46-9]|9[2-9])|"                              .
              "5(?:0[0-35-8]|1[024]|2[02-6]|3[025-7]|4[02-8]|5[2-68]|"       .
                  "6[014-6]|7[02-7]|8[0-9]|9[02-9])|"                        .
              "6(?:0[02-8]|1[02-8]|2[0-7]|3[02-9]|4[056])|"                  .
              "7(?:0[02-46-8]|1[2-8]|2[235-7]|3[02-57-9]|40|5[0-9]|"         .
                  "6[25-7]|7[2-57]|8[2-4])|"                                 .
              "8(?:0[0-8]|1[02356]|2[045]|3[2-6]|4[0-79]|5[2-8]|6[2-8]|"     .
                  "7[2-47-9]|8[0-9]|9[02-8])|"                               .
              "9(?:0[1-9]|1[0-9]|2[56]|3[2-4]|42|5[1-7]|6[24-7]|70))"        .

         "|9(?:0(?:0[0146-9]|1[0-6]|2[02-9]|3[02-8]|4[2-4]|5[02-8]|"         .
                  "6[2-4])|"                                                 .
              "1(?:0[0-578]|1[2-6]|2[235-7])|"                               .
              "2(?:0[013-5]|1[2-7]|2[035]|3[01]|4[02-9])|"                   .
              "3(?:0[014-68]|1[2-5]|2[0235-7])|"                             .
              "4(?:0[0-5]|1[0134]|2[2-8]|3[04-7]|4[2-5]|5[0-3]|6[2-9]|"      .
                  "7[0-35-9])|"                                              .
              "5(?:0[0-46-8]|1[02457]|2[3-7]|3[2-6]|4[235-8]|5[2-6]|"        .
                  "6[25]|73)|"                                               .
              "6(?:0[1246-8]|1[2-5]|2[0-2]|3[013]|4[23]|5[0-25-8]))"         .
    ")",


    #
    # Vatican City uses a single postal code, taken from the Italian
    # system for postal codes; and this code is shared with parts of Rome.
    #
    # Data from: http://download.geonames.org/export/zip/CH.zip
    #
   'Vatican City' => "(?k:00120)",

);

my %alternatives = (
    Australia    => [qw /Australian/],
    France       => [qw /French/],
    Germany      => [qw /German/],
);


while (my ($country, $zip) = each %zip) {
    my @names = ($country);
    push @names => @{$alternatives {$country}} if $alternatives {$country};
    foreach my $name (@names) {
        my $pat_name = $name eq "Denmark" && $] < 5.00503
                       ?   [zip => $name, qw /-country=/]
                       :   [zip => $name, qw /-prefix= -country=/];
        pattern name    => $pat_name,
                create  => sub {
                    my $pt  = _t $_ [1] {-prefix};

                    my $cn  = _c $country => $_ [1] {-country};
                    my $pfx = "(?:(?k:$cn)-)";

                    "(?k:$pfx$pt$zip)";
                },
                ;
    }
}


# Postal codes of the form 'DDDD LL', with F, I, O, Q, U and Y not
# used, SA, SD and SS unused combinations, and the first digit
# cannot be 0. No specific meaning to the letters or digits.
foreach my $country (qw /Netherlands Dutch/) {
    pattern name   => ['zip', $country => qw /-prefix= -country=/, "-sep= "],
            create => sub {
                my $pt  = _t $_ [1] {-prefix};

                # Unused letters: F, I, O, Q, U, Y.
                # Unused combinations: SA, SD, SS.
                my $num =  '[1-9][0-9]{3}';
                my $let =  '[A-EGHJ-NPRTVWXZ][A-EGHJ-NPRSTVWXZ]|' .
                           'S[BCEGHJ-NPRTVWXZ]';

                my $sep = __ $_ [1] {-sep};
                my $cn  = _c Netherlands => $_ [1] {-country};
                my $pfx = "(?:(?k:$cn)-)";

                "(?k:$pfx$pt(?k:(?k:$num)(?k:$sep)(?k:$let)))";
            },
            ;
}


# Postal codes of the form 'DDDDD' or 'DDDDD-DDDD'. All digits are used,
# none carry any specific meaning.
pattern name    => [qw /zip US -prefix= -country= -extended= -sep=-/],
        create  => sub {
            my $pt  = _t $_ [1] {-prefix};
            my $et  = _t $_ [1] {-extended};

            my $sep = __ $_ [1] {-sep};

            my $cn  = _c USA => $_ [1] {-country};
            my $pfx = "(?:(?k:$cn)-)";
            # my $zip = "(?k:[0-9]{5})";
            # my $ext = "(?:(?k:$sep)(?k:[0-9]{4}))";
            my $zip = "(?k:(?k:[0-9]{3})(?k:[0-9]{2}))";
            my $ext = "(?:(?k:$sep)(?k:(?k:[0-9]{2})(?k:[0-9]{2})))";

            "(?k:$pfx$pt(?k:$zip$ext$et))";
        },
        ;




#
# Postal codes are four digits, but not all combinations are used.
#
# Valid codes from:
#       https://postcode.auspost.com.au/free_display.html?id=1
# 
pattern name   => ['zip', 'Australia' => qw /-prefix= -country= -lax=/],
        create => sub {
            my $pt  = _t $_ [1] {-prefix};
            my $cn  = _c Australia => $_ [1] {-country};
            my $pfx = "(?:(?k:$cn)-)";
            my $lax = !defined $_ [1] {-lax} || $_ [1] {-lax};
            my $l0  = $lax ? "0?" : "0";   # Leading zero

            my $pat = "(?|" .
               "(?|1(?:2(?:15|2[05]|3[05]|40)|"                               .
                      "3(?:00|35|40|5[05]|60)|"                               .
                      "4(?:35|45|5[05]|6[056]|7[05]|8[015]|9[059])|"          .
                      "5(?:15|6[05]|70|85|9[05])|"                            .
                      "6(?:3[05]|40|55|60|7[05]|8[05])|"                      .
                      "7(?:0[01]|1[05]|30|5[05]|65|90)|"                      .
                      "8(?:0[05]|11|25|35|51|60|7[15]|85|90))"                .

                 "|2(?:0(?:0[0-246-9]|1[0-25-9]|[2-4][0-9]|5[0279]|"          .
                          "6[0-9]|7[0-79]|8[0-9]|9[02-79])|"                  .
                      "1(?:[0-2][0-9]|3[0-8]|4[0-8]|5[0-9]|6[0-8]|"           .
                          "7[0-9]|9[0-9])|"                                   .
                      "2(?:0[03-9]|1[0-46-9]|2[0-9]|3[0-4]|5[016-9]|"         .
                          "6[0-57]|78|8[0-79]|9[0-9])|"                       .
                      "3(?:0[02-9]|1[0-24-9]|2[0-9]|3[03-9]|4[0-8]|"          .
                          "5[0-9]|6[0159]|7[0-29]|8[0-26-8]|9[05-9])|"        .
                      "4(?:0[0-689]|1[015]|2[0-9]|3[019]|4[013-9]|"           .
                          "5[02-6]|6[02-69]|[78][0-9]|90)|"                   .
                      "5(?:0[02568]|1[5-9]|2[025-9]|3[03-9]|4[015689]|"       .
                          "5[015-9]|6[03-9]|7[0-9]|8[0-8]|9[04])|"            .
                      "6(?:0[0-9]|1[0-24-9]|2[0-9]|3[0-3]|4[0-9]|"            .
                          "5[0-35689]|6[0135689]|7[1258]|8[01])|"             .
                      "7(?:0[0-35-8]|1[0-7]|2[0-25-79]|3[0-9]|4[57-9]|"       .
                          "5[0-46-9]|6[0-35-9]|7[03-9]|8[02-7]|9[0-57-9])|"   .
                      "8(?:0[03-9]|1[078]|2[0-9]|3[0-689]|4[02-9]|5[02]|"     .
                          "6[4-9]|7[013-9]|80|9[089])|"                       .
                      "9(?:0[0-6]|1[1-4]))"                                   .
       
                 "|3(?:0(?:0[0-468]|1[0-35689]|2[0-9]|3[0-46-9]|[45][0-9]|"   .
                          "6[0-8]|7[0-689]|8[1-57-9]|9[013-79])|"             .
                      "1(?:0[1-9]|1[13-6]|2[1-9]|[34][0-9]|5[0-689]|"         .
                          "[6-9][0-9])|"                                      .
                      "2(?:0[0-24-7]|1[1-9]|2[0-8]|3[0-9]|4[0-39]|5[014]|"    .
                          "6[04-9]|7[0-9]|8[0-79]|9[2-4])|"                   .
                      "3(?:0[0-59]|1[0-2457-9]|2[1-589]|3[0-578]|4[0-25]|"    .
                          "5[0-7]|6[0134]|7[013-57-9]|8[014578]|9[0-356])|"   .
                      "4(?:0[0-279]|1[2-589]|2[0347-9]|3[0-578]|4[0-246-8]|"  .
                          "5[0138]|6[0-57-9]|7[2578]|8[02357-9]|9[01468])|"   .
                      "5(?:0[0-25-79]|1[25-8]|2[0-3579]|3[0137]|4[02469]|"    .
                          "5[0-24-9]|6[1-8]|7[0-3569]|8[013-689]|9[014-79])|" .
                      "6(?:0[78]|1[0246-9]|2[0-49]|3[0-9]|4[0134679]|"        .
                          "5[89]|6[0-69]|7[0-35-8]|8[2357-9]|9[01457-9])|"    .
                      "7(?:0[01457-9]|1[1-57-9]|2[02-8]|3[0235-9]|4[014679]|" .
                          "5[0-9]|6[0-7]|7[057-9]|8[1-35-9]|9[1-35-79])|"     .
                      "8(?:0[02-9]|1[02-68]|2[0-5]|3[1-35]|4[0-247]|"         .
                          "5[0-46-9]|6[02459]|7[013-58]|8[025-9]|9[0-3568])|" .
                      "9(?:0[02-49]|1[0-35689]|2[0-35-9]|3[01346-9]|4[0-6]|"  .
                          "5[01346-9]|6[024-7]|7[15-9]|8[01478]|9[0-256]))"   .

                 "|4(?:0(?:0[0-9]|1[0-47-9]|2[0-259]|3[0-24-7]|5[13-59]|"     .
                          "6[014-9]|7[02-8])|"                                .
                      "1(?:0[1-9]|1[0-9]|2[0-57-9]|3[0-3]|5[1-9]|6[013-59]|"  .
                          "7[0-489]|8[34])|"                                  .
                      "2(?:0[57-9]|[12][0-9]|30|7[0-25]|8[057])|"             .
                      "3(?:0[013-79]|1[0-3]|4[0-7]|5[02-9]|6[0-5]|7[0-8]|"    .
                          "8[0-578]|90)|"                                     .
                      "4(?:0[0-8]|1[0-35-9]|2[0-8]|5[45]|6[12578]|"           .
                          "7[0-2457-9]|8[0-26-9]|9[0-46-8])|"                 .
                      "5(?:0[0-9]|1[0-24-9]|2[01]|[56][0-9]|7[0-5]|8[01])|"   .
                      "6(?:0[01568]|1[0-5]|2[015-7]|30|5[059]|6[02]|"         .
                          "7[01346-8]|80|9[4579])|"                           .
                      "7(?:0[0-79]|1[0-9]|2[0-8]|3[0-35-9]|4[0-6]|5[013467]|" .
                          "9[89])|"                                           .
                      "8(?:0[02-9]|1[0-9]|2[0-589]|30|49|5[024-9]|"           .
                          "6[01589]|7[0-9]|8[0-8]|9[0-25]))"                  .

                 "|5(?:0(?:0[016-9]|1[0-9]|2[0-5]|3[1-57-9]|4[0-9]|5[0-2]|"   .
                          "6[1-9]|7[0-6]|8[1-9]|9[0-8])|"                     .
                      "1(?:0[6-9]|1[0-8]|2[015-7]|3[1-46-9]|4[0-24]|"         .
                          "[56][0-9]|7[0-4])|"                                .
                      "2(?:0[1-4]|1[0-4]|2[0-3]|3[1-8]|4[0-5]|5[0-69]|"       .
                          "6[0-9]|7[0-35-9]|80|9[01])|"                       .
                      "3(?:0[1-46-9]|1[01]|2[0-2]|3[0-3]|4[0-6]|5[0-7]|"      .
                          "60|7[1-4]|81)|"                                    .
                      "4(?:0[01]|1[0-9]|2[0-2]|3[1-4]|40|5[1-5]|6[0-24]|"     .
                          "7[0-3]|8[0-35]|9[0135])|"                          .
                      "5(?:0[12]|10|2[0-3]|40|5[024-68]|60|7[0-35-7]|"        .
                          "8[0-3])|"                                          .
                      "6(?:0[0-9]|3[0-3]|4[0-2]|5[0-5]|6[01]|7[01]|80|90)|"   .
                      "7(?:0[01]|1[09]|2[02-5]|3[0-4])|"                      .
                      "9(?:42|50))"                                           .

                 "|6(?:0(?:0[013-9]|1[0-24-9]|2[0-9]|3[0-8]|4[1-4]|"          .
                          "[56][0-9]|7[0-46-9]|8[1-4]|90)|"                   .
                      "1(?:0[0-9]|1[0-2]|2[1-6]|4[7-9]|[56][0-9]|"            .
                          "7[0-6]|8[0-2])|"                                   .
                      "2(?:0[7-9]|1[013-58]|2[013-9]|3[0-36769]|4[034]|"      .
                          "5[1-68]|6[02]|7[15]|8[0-24-68]|90)|"               .
                      "3(?:0[24689]|1[1-35-8]|2[0-46-8]|3[0-35-8]|4[1368]|"   .
                          "5[0-35-9]|6[1357-9]|7[0235]|8[3-6]|9[0-8])|"       .
                      "4(?:0[13579]|1[0-589]|2[0-9]|3[0-46-8]|4[0235-8]|"     .
                          "5[02]|6[0-35-8]|7[0235-79]|8[0457-9]|90)|"         .
                      "5(?:0[1-79]|1[0-9]|2[1258]|3[0-25-7]|5[68]|"           .
                          "6[0246-9]|7[1245])|"                               .
                      "6(?:0[35689]|1[2-46]|2[03578]|3[0-2589]|4[026])|"      .
                      "7(?:0[157]|1[0-468]|2[0-2568]|3[13]|4[03]|5[1348]|"    .
                          "6[025]|70|9[89])|"                                 .
                      "8(?:3[17-9]|4[0-9]|50|65|72|92)|"                      .
                      "9(?:0[1-79]|1[0-9]|2[0-69]|3[1-69]|4[1-7]|5[1-9]|"     .
                          "6[013-9]|7[09]|8[1-9]|9[0-27]))"                   .

                 "|7(?:0(?:0[0-24-9]|1[0-25-9]|2[0-7]|30|5[0-5])|"            .
                      "1(?:09|1[23679]|20|39|40|5[015]|6[23]|7[0-9]|"         .
                          "8[02-7]|90)|"                                      .
                      "2(?:09|1[0-6]|4[89]|5[02-9]|6[0-578]|7[05-7]|9[0-2])|" .
                      "3(?:0[0-7]|1[056]|2[0-25]|3[01])|"                     .
                      "4(?:6[6-9]|70))"                                       .

                 "|8(?:0(?:0[1-9]|1[0-2]))"                                   .

                 "|9726"                                                      .

                 #
                 # Place this last, in case the leading 0 is optional; if
                 # not, matching against "2001" would find "200".
                 #
                 "|$l0(?:200|8(?:0[014]|1[0-5]|2[0-289]|3[0-24-9]|4[05-7]|"   .
                               "5[0-4]|6[0-2]|7[0-5]|8[0156])|909)"           .
            "))";

            "(?k:$pfx$pt(?k:$pat))";
        }
        ;
    


# pattern name   => [qw /zip British/, "-sep= "],
#         create => sub {
#             my $sep     = $_ [1] -> {-sep};
# 
#             my $london  = '(?:EC[1-4]|WC[12]|S?W1)[A-Z]';
#             my $single  = '[BGLMS][0-9]{1,2}';
#             my $double  = '[A-Z]{2}[0-9]{1,2}';
# 
#             my $left    = "(?:$london|$single|$double)";
#             my $right   = '[0-9][ABD-HJLNP-UW-Z]{2}';
# 
#             "(?k:(?k:$left)(?k:$sep)(?k:$right))";
#         },
#         ;
# 
# pattern name   => [qw /zip Canadian/, "-sep= "],
#         create => sub {
#             my $sep     = $_ [1] -> {-sep};
# 
#             my $left    = '[A-Z][0-9][A-Z]';
#             my $right   = '[0-9][A-Z][0-9]';
# 
#             "(?k:(?k:$left)(?k:$sep)(?k:$right))";
#         },
#         ;


1;

__END__

=pod

=head1 NAME

Regexp::Common::zip -- provide regexes for postal codes.

=head1 SYNOPSIS

    use Regexp::Common qw /zip/;

    while (<>) {
        /^$RE{zip}{Netherlands}$/   and  print "Dutch postal code\n";
    }


=head1 DESCRIPTION

Please consult the manual of L<Regexp::Common> for a general description
of the works of this interface.

Do not use this module directly, but load it via I<Regexp::Common>.

This module offers patterns for zip or postal codes of many different
countries. They all have the form C<$RE{zip}{Country}[{options}]>.

The following common options are used:

=head2 C<{-prefix=[yes|no|allow]}> and C<{-country=PAT}>.

Postal codes can be prefixed with a country abbreviation. That is,
a Dutch postal code of B<1234 AB> can also be written as B<NL-1234 AB>.
By default, all the patterns will allow the prefixes. But this can be
changed with the C<-prefix> option. With C<-prefix=yes>, the returned
pattern requires a country prefix, while C<-prefix=no> disallows a
prefix. Any argument that doesn't start with a C<y> or a C<n> allows a
country prefix, but doesn't require them.

The prefixes used are, unfortunately, not always the same. Officially,
ISO country codes need to be used, but the usage of I<CEPT> codes (the
same ones as used on cars) is common too. By default, each postal code
will recognize a country prefix that's either the ISO standard or the
CEPT code. That is, German postal codes may prefixed with either C<DE>
or C<D>. The recognized prefix can be changed with the C<-country>
option, which takes a (sub)pattern as argument. The arguments C<iso>
and C<cept> are special, and indicate the language prefix should be the
ISO country code, or the CEPT code.

Examples:
 /$RE{zip}{Netherlands}/;
           # Matches '1234 AB' and 'NL-1234 AB'.
 /$RE{zip}{Netherlands}{-prefix => 'no'}/;
           # Matches '1234 AB' but not 'NL-1234 AB'.
 /$RE{zip}{Netherlands}{-prefix => 'yes'}/;
           # Matches 'NL-1234 AB' but not '1234 AB'.

 /$RE{zip}{Germany}/;
           # Matches 'DE-12345' and 'D-12345'.
 /$RE{zip}{Germany}{-country => 'iso'}/; 
           # Matches 'DE-12345' but not 'D-12345'.
 /$RE{zip}{Germany}{-country => 'cept'}/;
           # Matches 'D-12345' but not 'DE-12345'.
 /$RE{zip}{Germany}{-country => 'GER'}/;
           # Matches 'GER-12345'.

=head2 C<{-sep=PAT}>

Some countries have postal codes that consist of two parts. Typically
there is an official way of separating those parts; but in practise
people tend to use different separators. For instance, if the official
way to separate parts is to use a space, it happens that the space is
left off. The C<-sep> option can be given a pattern as argument which
indicates what to use as a separator between the parts.

Examples:
 /$RE{zip}{Netherlands}/;
           # Matches '1234 AB' but not '1234AB'.
 /$RE{zip}{Netherlands}{-sep => '\s*'}/;
           # Matches '1234 AB' and '1234AB'.

=head2 C<$RE{zip}{Australia}{-lax}>

Returns a pattern that recognizes Australian postal codes. Australian
postal codes consist of four digits; the first two digits, which range
from '10' to '97', indicate the state, although there are exceptions.
Territories use '02' or '08' as starting digits. '0909' is the only 
postal code starting with '09' - this is the postal code for the
Northern Territory University). The (optional) country
prefixes are I<AU> (ISO country code) and I<AUS> (CEPT code).

It the past, it was claimed that for postal codes starting with a 0,
the leading 0 may be omitted, and up to (and including) version
2016060201, the leading 0 was optional. But there doesn't seem be
solid evidence the leading 0 is optional. So, we now require there
always to be four digit -- unless the C<< {-lax} >> option is given,
then a possibly leading 0 is optional.

Regexp::Common 2.107 and before used C<$RE{zip}{Australian}>. This is
still supported.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

As of version 2016060201, no C<< $4 >> or C<< $5 >> will be set.

=head2 C<< $RE {zip} {Austria} >>

Returns a pattern which recognizes Austrian postal codes. Austrian postal
codes consists of 4 digits, but not all possibilities are used. This
pattern matches the postal codes in use. The (optional) country prefixes
are I<AT> (ISO country code) and I<AUT> (CEPT code).

If C<< {-keep} >> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country code prefix.

=back

=head2 C<$RE{zip}{Belgium}>

Returns a pattern than recognizes Belgian postal codes. Belgian postal
codes consist of 4 digits, of which the first indicates the province.
The (optional) country prefixes are I<BE> (ISO country code) and
I<B> (CEPT code).

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back



=head2 C<$RE{zip}{Denmark}>

Returns a pattern that recognizes Danish postal codes. Danish postal
codes consist of four numbers; the first digit indicates the
distribution region, the second the distribution district. The
(optional) country prefix is I<DK>, which is both the ISO country
code and the CEPT code.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

Danish postal codes will not start with 39. Postal codes of the form
39XX are reserved from Greenland; the pattern for Danish postal codes
will not recognize them.


=head2 C<$RE{zip}{France}>

Returns a pattern that recognizes French postal codes. French postal
codes consist of five numbers; the first two numbers, which range
from '01' to '98', indicate the department. The (optional) country
prefixes are I<FR> (ISO country code) and I<F> (CEPT code).
Regexp::Common 2.107 and before used C<$RE{zip}{French}>. This is
still supported.

Monaco uses postal codes which are part of the numbering system used
by the French postal code system; their numbers start with 980. These
numbers are C<< not >> recognized by this pattern.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

=head2 C<$RE{zip}{Germany}>

Returns a pattern that recognizes German postal codes. German postal
codes consist of five numbers; the first two numbers indicating a
wider postal area, the last three digits a postal district.
The (optional) country prefixes are I<DE> (ISO country code) and
I<D> (CEPT code).
Regexp::Common 2.107 and before used C<$RE{zip}{German}>. This is
still supported.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back


=head2 C<$RE{zip}{Greenland}>

Returns a pattern that recognizes postal codes from Greenland.
Greenland, uses the Danish postal codes system. Postal codes starting
with 39 are reserved for Greenland, and all Greenlandic postal codes
start with 39. Except the postal code for Santa. He uses 2412.

The (optional) country prefix is I<GL>, which is use both as
the ISO country code and the CEPT code. Earlier versions used
I<DK> as the prefix.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

=head2 C<$RE{zip}{Italy}>

Returns a pattern recognizing Italian postal codes. Italian postal
codes consist of 5 digits. The first digit indicates the region, the
second the province. The third digit is odd for province capitals,
and even for the province itself. The fourth digit indicates the
route, and the fifth a place on the route (0 for small places, 
alphabetically for the rest).

Codes starting with 4789 are postal codes for San Marino; they are
not recognized by the pattern. Use C<< $RE {zip} {'San Marino'} >>
instead.

The country prefix is either I<IT> (the ISO country code), or
I<I> (the CEPT code).

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

=head2 C<< $RE {zip} {Liechtenstein} >>

Returns a pattern which recognizes postal codes used in Liechtenstein.
Liechtenstein uses postal codes from the Swiss postal code system.
This system uses four digits. Postal codes which start with 94, and
use 8 or 9 as a third digit are postal codes for Liechtenstein.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

The ISO country prefix is I<< LI >>, the CEPT country prefix is I<< LIE >>.

=head2 C<< $RE {zip {Monaco} >>

Returns a pattern for postal codes used in Monaco. Monaco uses a range
from the system used in France. They are 5 digits, starting with I<< 980 >>.
The number I<< 98000 >> is used for physical addresses. Numbers ending
in C<< 01 >> to C<< 99 >> are used for special deliveries.

The ISO country code is I<< MC >>.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

=head2 C<$RE{zip}{Netherlands}>

Returns a pattern that recognizes Dutch postal codes. Dutch postal
codes consist of 4 digits and 2 letters, separated by a space.
The separator can be changed using the C<{-sep}> option, as discussed
above. The (optional) country prefix is I<NL>, which is both the 
ISO country code and the CEPT code. Regexp::Common 2.107 and earlier
used C<$RE{zip}{Dutch}>. This is still supported.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=item $4

The digits part of the postal code.

=item $5

The separator between the digits and the letters.

=item $6 

The letters part of the postal code.

=back

=head2 C<< $RE{zip}{Norway} >>

Returns a pattern that recognizes Norwegian postal codes. Norwegian
postal codes consist of four digits.

The country prefix is either I<NO> (the ISO country code), or
I<N> (the CEPT code).

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

=head2 C<< $RE {zip} {'San Marino'} >>

Postal codes of San Marino use a slice from the Italian postal codes.
Any code starting 4789, followed by another digit belongs to San Marino.

The country prefix for San Marino is I<< SM >>.


If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back


=head2 C<< $RE{zip}{Spain} >>

Returns a pattern that recognizes Spanish postal codes. Spanish postal
codes consist of 5 digits. The first 2 indicate one of Spain's fifties
provinces (in alphabetical order), starting with C<00>. The third digit
indicates a main city or the main delivery rounds. The last two digits
are the delivery area, secondary delivery route or a link to rural areas.

The country prefix is either I<ES> (the ISO country code), or
I<E> (the CEPT code).

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=item $4

The two digits indicating the province.

=item $5

The digit indicating the main city or main delivery route.

=item $6

The digits indicating the delivery area, secondary delivery route
or a link to rural areas.

=back

=head2 C<< $RE {zip} {Switzerland} >>

Returns a pattern that recognizes Swiss postal codes. Swiss postal
codes consist of 4 digits, but not all combinations are used. Postal
codes starting with 948 and 949 are for location in Liechtenstein,
and will not be recognized by the pattern for Swiss postal codes.
Use C<< $RE {zip} {Liechtenstein} >> for those.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

The country prefix is I<CH>, for both the ISO and CEPT prefixes.



=head2 C<< $RE{zip}{US}{-extended => [yes|no|allow]} >>

Returns a pattern that recognizes US zip codes. US zip codes consist
of 5 digits, with an optional 4 digit extension. By default, extensions
are allowed, but not required. This can be influenced by the 
C<-extended> option. If its argument starts with a C<y>,
extensions are required; if the argument starts with a C<n>,
extensions will not be recognized. If an extension is used, a dash
is used to separate the main part from the extension, but this can
be changed with the C<-sep> option.

The country prefix is either I<US> (the ISO country code), or
I<USA> (the CEPT code).

If C<{-keep}> is being used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=item $4

The first 5 digits of the postal code.

=item $5

The first three digits of the postal code, indicating a sectional
center or a large city. New in Regexp::Common 2.119.

=item $6

The last 2 digits of the 5 digit part of the postal code, indicating
a post office facility or delivery area. New in Regexp::Common 2.119.

=item $7

The separator between the 5 digit part and the 4 digit part. Up to 
Regexp::Common 2.118, this used to be $5.

=item $8

The 4 digit part of the postal code (if any). Up to Regexp::Common 2.118,
this used to be $6.

=item $9

The first two digits of the 4 digit part of the postal code, indicating
a sector, or several blocks. New in Regexp::Common 2.119.

=item $10

The last two digits of the 4 digit part of the postal code, indicating
a segment or one side of a street. New in Regexp::Common 2.119.

=back

=head3 Questions

=over 4

=item

Can the 5 digit part of the zip code (in theory) start with 000?

=item

Can the 5 digit part of the zip code (in theory) end with 00?

=item

Can the 4 digit part of the zip code (in theory) start with 00?

=item

Can the 4 digit part of the zip code (in theory) end with 00?

=back

=head2 C<< $RE {zip} {'Vatican City'} >>

Vatican City uses a single postal code; taken from the Italian 
system of postal codes, and sharing the single code with a part
of Rome.

If C<{-keep}> is used, the following variables will be set:

=over 4

=item $1

The entire postal code.

=item $2

The country code prefix.

=item $3

The postal code without the country prefix.

=back

The country prefix for Vatican City is C<< VA >>.

=head1 SEE ALSO

L<Regexp::Common> for a general description of how to use this interface.

=over 4

=item L<http://www.columbia.edu/kermit/postal.html>

Frank's compulsive guide to postal addresses.

=item L<http://www.upu.int/post_code/en/addressing_formats_guide.shtml>

Postal addressing systems.

=item L<http://www.uni-koeln.de/~arcd2/33e.htm>

Postal code information.

=item L<http://www.grcdi.nl/linkspc.htm>

Links to Postcode Pages.

=item L<https://postcode.auspost.com.au/free_display.html?id=1>

All Australian postal codes in use.

=item L<http://hdusps.esecurecare.net/cgi-bin/hdusps.cfg/php/enduser/std_adp.php?p_faqid=1014>

Information about US postal codes.

=item L<http://en.wikipedia.org/wiki/Postal_code>

=item L<http://download.geonames.org/export/zip/>

Lots of zip files with active postal codes.

=item L<http://postal-codes.findthedata.com/>

Find postal codes.

=back

=head1 AUTHORS

Damian Conway S<(I<damian@conway.org>)> and
Abigail S<(I<regexp-common@abigail.freedom.nl>)>.

=head1 MAINTENANCE

This package is maintained by Abigail S<(I<regexp-common@abigail.freedom.nl>)>.

=head1 BUGS AND IRRITATIONS

Zip codes for most countries are missing.
Send them in to I<regexp-common@abigail.freedom.nl>.

=head1 LICENSE and COPYRIGHT

This software is Copyright (c) 2001 - 2024, Damian Conway and Abigail.

This module is free software, and maybe used under any of the following
licenses:

 1) The Perl Artistic License.     See the file COPYRIGHT.AL.
 2) The Perl Artistic License 2.0. See the file COPYRIGHT.AL2.
 3) The BSD License.               See the file COPYRIGHT.BSD.
 4) The MIT License.               See the file COPYRIGHT.MIT.

=cut
